<?php
require_once '../includes/config.php';
require_once '../includes/functions.php';

// Verifica se o site está ativo
$config = getSiteConfig();
if (!isset($config['site_ativo']) || $config['site_ativo'] != 1) {
    header('Location: site_desativado.php');
    exit;
}

// Debug temporário
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Inicializa as variáveis
$logo = '';
$categorias = [];
$produtos = [];
$errors = [];
$config_empresa = [];
$config_redes = [];
$cores = [];
$tamanhos = [];
$slide_ativo = false; // Inicializa como falso

// Define o título da página
$pageTitle = 'Loja';

// Inclui o header
require_once 'header.php';

// Função para verificar se uma coluna existe na tabela
function columnExists($conn, $table, $column) {
    try {
        $sql = "SHOW COLUMNS FROM `{$table}` WHERE Field = '{$column}'";
        $result = $conn->query($sql);
        return $result && $result->rowCount() > 0;
    } catch(PDOException $e) {
        error_log("Erro ao verificar coluna: " . $e->getMessage());
        return false;
    }
}

// Pegar categoria da URL se existir
$categoria_id = isset($_GET['categoria']) ? (int)$_GET['categoria'] : null;

try {
    $db = getConnection();
    
    // Verifica a conexão
    if (!$db) {
        throw new Exception("Falha ao conectar ao banco de dados");
    }
    
    // Debug da estrutura do banco
    $stmt = $db->query("SHOW TABLES");
    $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
    error_log('DEBUG - Tabelas no banco: ' . print_r($tables, true));
    
    // Verifica se as tabelas necessárias existem
    $required_tables = ['produtos', 'categorias', 'config_empresa', 'config_redes_sociais'];
    foreach ($required_tables as $table) {
        if (!in_array($table, $tables)) {
            throw new Exception("Tabela '$table' não encontrada no banco de dados");
        }
    }
    
    // Verifica e cria as colunas se não existirem
    if (!columnExists($db, 'produtos', 'cor')) {
        $db->exec("ALTER TABLE produtos ADD COLUMN cor VARCHAR(50) DEFAULT NULL COMMENT 'Cor do produto'");
    }
    if (!columnExists($db, 'produtos', 'tamanho')) {
        $db->exec("ALTER TABLE produtos ADD COLUMN tamanho VARCHAR(50) DEFAULT NULL COMMENT 'Tamanho do produto'");
    }
    
    // Buscar dados da empresa
    $stmt = $db->query("SELECT * FROM config_empresa LIMIT 1");
    $config_empresa = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Buscar dados das redes sociais
    $stmt = $db->query("SELECT * FROM config_redes_sociais LIMIT 1");
    $config_redes = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($config_redes) {
        // Adiciona as redes sociais ao array de configurações da empresa
        $config_empresa['instagram'] = $config_redes['instagram'];
        $config_empresa['facebook'] = $config_redes['facebook'];
        $config_empresa['tiktok'] = $config_redes['tiktok'];
        $config_empresa['linkedin'] = $config_redes['linkedin'];
    }
    
    // Busca configurações do site
    $stmt = $db->query("SELECT * FROM config_site LIMIT 1");
    $config_site = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Define se o slide está ativo
    $slide_ativo = isset($config_site['slide_ativo']) && $config_site['slide_ativo'] == 1;
    $slide_texto_principal = $config_site['slide_texto_principal'] ?? '';
    $slide_texto_secundario = $config_site['slide_texto_secundario'] ?? '';
    
    if ($config_empresa) {
        error_log("DEBUG - Configurações completas: " . print_r($config_empresa, true));
        $logo = $config_empresa['logo'] ?? '';
        if (!empty($logo)) {
            $logo = '../' . ltrim($logo, '/');
            if (!file_exists($logo)) {
                error_log("DEBUG - Logo não encontrada: $logo");
                $logo = '';
            }
        }
    } else {
        error_log("DEBUG - Nenhuma configuração de empresa encontrada");
    }
    
    // Buscar categorias
    $stmt = $db->query("SELECT * FROM categorias WHERE deleted_at IS NULL ORDER BY nome");
    $categorias = $stmt->fetchAll(PDO::FETCH_ASSOC);
    if (empty($categorias)) {
        error_log("DEBUG - Nenhuma categoria encontrada");
    }
    
    // Buscar cores e tamanhos únicos
    $coresUnicas = [];
    $tamanhosUnicos = [];

    // Buscar cores
    $stmt = $db->query("SELECT DISTINCT cor FROM produtos WHERE deleted_at IS NULL AND cor IS NOT NULL AND cor != '' ORDER BY cor");
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $coresUnicas[] = trim($row['cor']);
    }

    // Buscar tamanhos da tabela produto_tamanhos
    $stmt = $db->query("
        SELECT DISTINCT pt.tamanho 
        FROM produto_tamanhos pt 
        INNER JOIN produtos p ON p.id = pt.produto_id 
        WHERE p.deleted_at IS NULL 
        AND pt.tamanho IS NOT NULL 
        AND pt.tamanho != '' 
        ORDER BY pt.tamanho
    ");
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        if (!empty($row['tamanho'])) {
            $tamanhosUnicos[] = trim($row['tamanho']);
        }
    }

    // Remove duplicatas e ordena os tamanhos
    $tamanhosUnicos = array_unique($tamanhosUnicos);
    sort($tamanhosUnicos);

    // Buscar produtos com seus tamanhos
    $sql = "SELECT p.*, c.nome as categoria_nome,
                   COALESCE(p.valor, 0.00) as preco,
                   GROUP_CONCAT(pt.tamanho) as tamanhos
            FROM produtos p 
            LEFT JOIN categorias c ON p.categoria_id = c.id 
            LEFT JOIN produto_tamanhos pt ON p.id = pt.produto_id
            WHERE p.deleted_at IS NULL";
    
    if ($categoria_id) {
        $sql .= " AND p.categoria_id = :categoria_id";
    }
    
    $sql .= " GROUP BY p.id ORDER BY p.id DESC";
    
    $stmt = $db->prepare($sql);
    if ($categoria_id) {
        $stmt->bindParam(':categoria_id', $categoria_id, PDO::PARAM_INT);
    }
    $stmt->execute();
    $produtos = $stmt->fetchAll(PDO::FETCH_ASSOC);
    if (empty($produtos)) {
        error_log("DEBUG - Nenhum produto encontrado");
    }
    
} catch (Exception $e) {
    error_log('ERRO: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
    $errors[] = $e->getMessage();
}

// Debug final
error_log('DEBUG - Status final:');
error_log('- Logo: ' . ($logo ? 'Encontrada' : 'Não encontrada'));
error_log('- Categorias: ' . count($categorias));
error_log('- Produtos: ' . count($produtos));
error_log('- Erros: ' . count($errors));

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../favicon.png">
    <link rel="shortcut icon" type="image/png" href="../favicon.png">
    <title><?php echo formatPageTitle($pageTitle); ?></title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Bootstrap Icons -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.1/font/bootstrap-icons.css" rel="stylesheet">
    <!-- Custom CSS -->
    <link rel="stylesheet" href="../assets/css/theme.css">
    <link rel="stylesheet" href="../assets/css/loja.css">
</head>
<body>
    <?php if (!empty($errors)): ?>
    <div class="alert alert-danger" role="alert">
        <ul class="mb-0">
            <?php foreach ($errors as $error): ?>
                <li><?php echo htmlspecialchars($error); ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
    <?php endif; ?>
    <!-- Navbar -->
    <nav class="navbar navbar-expand-lg navbar-light bg-white fixed-top">
        <div class="container">
            <!-- Logo -->
            <a class="navbar-brand" href="index.php">
                <?php if (!empty($logo)): ?>
                    <img src="<?php echo $logo; ?>" alt="Logo" class="img-fluid">
                <?php else: ?>
                    <span class="h4 mb-0"><?php echo htmlspecialchars($config_empresa['nome'] ?? 'Minha Loja'); ?></span>
                <?php endif; ?>
            </a>
            
            <!-- Toggle Button -->
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            
            <!-- Menu Items -->
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav ms-auto">
                    <li class="nav-item">
                        <a class="nav-link <?php echo !$categoria_id ? 'active' : ''; ?>" href="index.php">Todos</a>
                    </li>
                    <?php foreach ($categorias as $categoria): ?>
                    <li class="nav-item">
                        <a class="nav-link <?php echo $categoria_id == $categoria['id'] ? 'active' : ''; ?>" 
                           href="?categoria=<?php echo $categoria['id']; ?>">
                            <?php echo htmlspecialchars($categoria['nome']); ?>
                        </a>
                    </li>
                    <?php endforeach; ?>
                    
                    <!-- Carrinho -->
                    <li class="nav-item ms-lg-3">
                        <a class="nav-link cart-button" href="#" id="cartButton">
                            <i class="bi bi-cart3"></i>
                            <span class="cart-count">0</span>
                        </a>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <!-- Content -->
    <main>
        <?php if ($slide_ativo): ?>
        <!-- Hero Slider -->
        <div id="heroSlider" class="carousel slide" data-bs-ride="carousel">
            <div class="carousel-inner">
                <div class="carousel-item active">
                    <div class="hero-slide" <?php if (!empty($config_site['slide_image'])): ?>style="background-image: url('<?php echo htmlspecialchars('../' . $config_site['slide_image']); ?>');"<?php endif; ?>>
                        <div class="container">
                            <div class="hero-content">
                                <h2><?php echo htmlspecialchars($slide_texto_principal); ?></h2>
                                <p><?php echo htmlspecialchars($slide_texto_secundario); ?></p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Carousel Controls -->
            <button class="carousel-control-prev" type="button" data-bs-target="#heroSlider" data-bs-slide="prev">
                <span class="carousel-control-prev-icon" aria-hidden="true"></span>
                <span class="visually-hidden">Anterior</span>
            </button>
            <button class="carousel-control-next" type="button" data-bs-target="#heroSlider" data-bs-slide="next">
                <span class="carousel-control-next-icon" aria-hidden="true"></span>
                <span class="visually-hidden">Próximo</span>
            </button>
        </div>
        <?php endif; ?>
        <!-- Products Section -->
        <div class="container py-4">
            <div class="row">
                <!-- Filtros Laterais -->
                <div class="col-md-3">
                    <div class="filter-section mb-4">
                        <h5 class="filter-title">Filtros</h5>
                        
                        <!-- Filtro de Cor -->
                        <div class="filter-section mb-4">
                            <h6 class="mb-3">Cores</h6>
                            <?php if (empty($coresUnicas)): ?>
                                <p class="text-muted">Nenhuma cor disponível</p>
                            <?php else: ?>
                                <div class="d-flex flex-column gap-2">
                                    <?php foreach ($coresUnicas as $cor): ?>
                                        <div class="form-check">
                                            <input class="form-check-input filter-cor" 
                                                   type="checkbox" 
                                                   value="<?php echo htmlspecialchars($cor); ?>" 
                                                   id="cor-<?php echo htmlspecialchars($cor); ?>">
                                            <label class="form-check-label" for="cor-<?php echo htmlspecialchars($cor); ?>">
                                                <?php echo htmlspecialchars($cor); ?>
                                            </label>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>

                        <!-- Filtro de Tamanhos -->
                        <div class="filter-section mb-4">
                            <h6 class="mb-3">Tamanhos</h6>
                            <?php if (empty($tamanhosUnicos)): ?>
                                <p class="text-muted">Nenhum tamanho disponível</p>
                            <?php else: ?>
                                <div class="d-flex flex-column gap-2">
                                    <?php foreach ($tamanhosUnicos as $tamanho): ?>
                                        <div class="form-check">
                                            <input class="form-check-input filter-tamanho" 
                                                   type="checkbox" 
                                                   value="<?php echo htmlspecialchars($tamanho); ?>" 
                                                   id="tamanho-<?php echo htmlspecialchars($tamanho); ?>">
                                            <label class="form-check-label" for="tamanho-<?php echo htmlspecialchars($tamanho); ?>">
                                                <?php echo htmlspecialchars($tamanho); ?>
                                            </label>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>

                        <!-- Filtro de Valor -->
                        <div class="">
                            <h6>Faixa de Preço</h6>
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" value="0-50" id="preco-1">
                                <label class="form-check-label" for="preco-1">Até R$ 50</label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" value="50-100" id="preco-2">
                                <label class="form-check-label" for="preco-2">R$ 50 - R$ 100</label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" value="100-200" id="preco-3">
                                <label class="form-check-label" for="preco-3">R$ 100 - R$ 200</label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" value="200+" id="preco-4">
                                <label class="form-check-label" for="preco-4">Acima de R$ 200</label>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Grid de Produtos -->
                <div class="col-md-9">
                    <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4">
                        <?php foreach ($produtos as $produto): 
                            // Verifica e ajusta o caminho da imagem
                            $imagem = '';
                            if (!empty($produto['imagem'])) {
                                $imagemPath = ltrim($produto['imagem'], './');
                                $imagem = '../uploads/produtos/' . basename($imagemPath);
                                
                                // Adiciona timestamp para evitar cache
                                if (file_exists($imagem)) {
                                    $timestamp = filemtime($imagem);
                                    $imagem .= '?v=' . $timestamp;
                                }
                            }
                            
                            // Prepara os tamanhos do produto
                            $tamanhosProduto = !empty($produto['tamanhos']) ? explode(',', $produto['tamanhos']) : [];
                            $tamanhosProdutoString = implode(',', $tamanhosProduto);
                        ?>
                            <div class="col">
                                <div class="card h-100 product-card" 
                                     data-cor="<?php echo htmlspecialchars($produto['cor'] ?? ''); ?>"
                                     data-tamanhos="<?php echo htmlspecialchars($tamanhosProdutoString); ?>"
                                     data-preco="<?php echo htmlspecialchars($produto['valor'] ?? '0'); ?>">
                                    
                                    <div class="card-img-wrapper">
                                        <?php if (!empty($imagem) && file_exists(str_replace('?v=' . $timestamp, '', $imagem))): ?>
                                            <img src="<?php echo htmlspecialchars($imagem); ?>" 
                                                 class="card-img-top" 
                                                 data-image-path="<?php echo htmlspecialchars(str_replace('../', '', $imagem)); ?>"
                                                 alt="<?php echo htmlspecialchars($produto['nome']); ?>">
                                        <?php else: ?>
                                            <img src="../assets/img/no-image.jpg" 
                                                 class="card-img-top" 
                                                 data-image-path="assets/img/no-image.jpg"
                                                 alt="Imagem não disponível">
                                        <?php endif; ?>
                                    </div>

                                    <div class="card-body">
                                        <h5 class="card-title"><?php echo htmlspecialchars($produto['nome']); ?></h5>
                                        
                                        <?php if (!empty($produto['categoria_nome'])): ?>
                                            <p class="card-text category">
                                                <small class="text-muted">
                                                    <?php echo htmlspecialchars($produto['categoria_nome']); ?>
                                                </small>
                                            </p>
                                        <?php endif; ?>

                                        <?php if (!empty($tamanhosProduto)): ?>
                                            <p class="card-text sizes">
                                                <small>Tamanhos: <?php echo htmlspecialchars(implode(', ', $tamanhosProduto)); ?></small>
                                            </p>
                                        <?php endif; ?>

                                        <div class="price-cart-wrapper">
                                            <p class="product-price">
                                                <?php 
                                                $preco = floatval($produto['valor']);
                                                echo 'R$ ' . number_format($preco, 2, ',', '.');
                                                ?>
                                            </p>
                                            <button class="btn btn-add-cart" 
                                                    data-product-id="<?php echo $produto['id']; ?>"
                                                    title="Adicionar ao carrinho">
                                                <i class="bi bi-cart-plus"></i>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <?php include 'footer.php'; ?>

    <!-- Toast Container -->
    <div class="position-fixed bottom-0 end-0 p-3" style="z-index: 11">
        <div id="liveToast" class="toast" role="alert" aria-live="assertive" aria-atomic="true">
            <div class="toast-header">
                <strong class="me-auto">Notificação</strong>
                <button type="button" class="btn-close" data-bs-dismiss="toast" aria-label="Close"></button>
            </div>
            <div class="toast-body"></div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Inicializa o carrinho
            const cartCountElement = document.querySelector('.cart-count');
            
            // Função para atualizar o contador do carrinho
            function updateCartCount() {
                const cartItems = JSON.parse(localStorage.getItem('cartItems') || '[]');
                const count = cartItems.reduce((sum, item) => sum + item.quantity, 0);
                cartCountElement.textContent = count;
            }
            
            // Atualiza o contador ao carregar a página
            updateCartCount();
            
            // Link do carrinho
            document.getElementById('cartButton').addEventListener('click', function(e) {
                e.preventDefault();
                window.location.href = 'carrinho.php';
            });
            
            // Adiciona evento de clique aos botões de adicionar ao carrinho
            const addToCartButtons = document.querySelectorAll('.btn-add-cart');
            addToCartButtons.forEach(button => {
                button.addEventListener('click', function() {
                    const productId = this.dataset.productId;
                    const card = this.closest('.product-card');
                    const imgElement = card.querySelector('.card-img-top');
                    
                    // Obtém os dados do produto
                    const product = {
                        id: productId,
                        name: card.querySelector('.card-title').textContent,
                        price: parseFloat(card.dataset.preco),
                        category: card.querySelector('.category .text-muted')?.textContent.trim() || '',
                        image: imgElement.getAttribute('data-image-path') || imgElement.src, // Usa o caminho da imagem do data attribute
                        quantity: 1
                    };
                    
                    // Recupera itens existentes do carrinho
                    const cartItems = JSON.parse(localStorage.getItem('cartItems') || '[]');
                    
                    // Verifica se o produto já está no carrinho
                    const existingItem = cartItems.find(item => item.id === productId);
                    if (existingItem) {
                        existingItem.quantity += 1;
                    } else {
                        cartItems.push(product);
                    }
                    
                    // Salva no localStorage
                    localStorage.setItem('cartItems', JSON.stringify(cartItems));
                    
                    // Atualiza o contador
                    updateCartCount();
                    
                    // Mostra toast de confirmação
                    const toast = new bootstrap.Toast(document.getElementById('liveToast'));
                    const toastBody = document.querySelector('.toast-body');
                    toastBody.textContent = 'Produto adicionado ao carrinho!';
                    toast.show();
                });
            });
            
            // Adiciona padding-top ao body para compensar a navbar fixa
            document.body.style.paddingTop = document.querySelector('.navbar').offsetHeight + 'px';
            
            // Código existente dos filtros
            const filtrosCor = document.querySelectorAll('.filter-cor');
            const filtrosTamanho = document.querySelectorAll('.filter-tamanho');
            const filtrosPreco = document.querySelectorAll('input[id^="preco-"]');
            
            // Função para atualizar a visibilidade dos produtos
            function atualizarProdutos() {
                const coresSelecionadas = Array.from(filtrosCor)
                    .filter(cb => cb.checked)
                    .map(cb => cb.value);
                    
                const tamanhosSelecionados = Array.from(filtrosTamanho)
                    .filter(cb => cb.checked)
                    .map(cb => cb.value);
                    
                const precosSelecionados = Array.from(filtrosPreco)
                    .filter(cb => cb.checked)
                    .map(cb => cb.value);
                
                // Seleciona todos os cards de produto
                const produtos = document.querySelectorAll('.product-card');
                let produtosVisiveis = 0;
                
                produtos.forEach(produto => {
                    let mostrarProduto = true;
                    
                    // Verifica filtro de cor
                    if (coresSelecionadas.length > 0) {
                        const corProduto = produto.dataset.cor;
                        if (!coresSelecionadas.includes(corProduto)) {
                            mostrarProduto = false;
                        }
                    }
                    
                    // Verifica filtro de tamanho
                    if (tamanhosSelecionados.length > 0 && mostrarProduto) {
                        const tamanhosProduto = produto.dataset.tamanhos.split(',');
                        const temTamanhoSelecionado = tamanhosSelecionados.some(tamanho => 
                            tamanhosProduto.includes(tamanho)
                        );
                        if (!temTamanhoSelecionado) {
                            mostrarProduto = false;
                        }
                    }
                    
                    // Verifica filtro de preço
                    if (precosSelecionados.length > 0 && mostrarProduto) {
                        const precoProduto = parseFloat(produto.dataset.preco);
                        const dentroDaFaixa = precosSelecionados.some(faixa => {
                            const [min, max] = faixa.split('-').map(Number);
                            return precoProduto >= min && (max ? precoProduto <= max : true);
                        });
                        if (!dentroDaFaixa) {
                            mostrarProduto = false;
                        }
                    }
                    
                    // Atualiza a visibilidade do produto
                    const colElement = produto.closest('.col');
                    if (mostrarProduto) {
                        colElement.style.display = '';
                        produtosVisiveis++;
                    } else {
                        colElement.style.display = 'none';
                    }
                });
                
                // Verifica se há produtos visíveis
                const mensagemNenhumProduto = document.getElementById('nenhumProduto');
                const containerProdutos = document.querySelector('.row.row-cols-1');
                
                if (produtosVisiveis === 0 && (coresSelecionadas.length > 0 || tamanhosSelecionados.length > 0 || precosSelecionados.length > 0)) {
                    if (!mensagemNenhumProduto) {
                        const mensagem = document.createElement('div');
                        mensagem.id = 'nenhumProduto';
                        mensagem.className = 'alert mt-3';
                        mensagem.style.backgroundColor = 'var(--cor-principal)';
                        mensagem.style.color = 'var(--cor-terciaria)';
                        mensagem.style.border = 'none';
                        mensagem.style.borderRadius = '4px';
                        mensagem.style.padding = '15px 20px';
                        mensagem.textContent = 'Nenhum produto encontrado com os filtros selecionados.';
                        containerProdutos.appendChild(mensagem);
                    }
                } else if (mensagemNenhumProduto) {
                    mensagemNenhumProduto.remove();
                }
            }
            
            // Adiciona listeners para todos os checkboxes
            filtrosCor.forEach(checkbox => {
                checkbox.addEventListener('change', atualizarProdutos);
            });
            
            filtrosTamanho.forEach(checkbox => {
                checkbox.addEventListener('change', atualizarProdutos);
            });
            
            filtrosPreco.forEach(checkbox => {
                checkbox.addEventListener('change', atualizarProdutos);
            });
        });
    </script>
</body>
</html>
