<?php
require_once '../includes/config.php';
require_once '../includes/functions.php';

// Inicializa variáveis
$errors = [];
$config_empresa = [];
$total = 0;

try {
    $db = getConnection();
    
    // Buscar dados da empresa
    $stmt = $db->query("SELECT * FROM config_empresa LIMIT 1");
    $config_empresa = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($config_empresa) {
        $logo = $config_empresa['logo'] ?? '';
        if (!empty($logo)) {
            $logo = '../' . ltrim($logo, '/');
            if (!file_exists($logo)) {
                $logo = '';
            }
        }
        
        // Pega o número do celular e remove formatação
        $celular = $config_empresa['celular'] ?? '';
        $whatsapp = preg_replace('/[^0-9]/', '', $celular);
        
        // Debug
        error_log('Celular original: ' . $celular);
        error_log('WhatsApp formatado: ' . $whatsapp);
    }
} catch (Exception $e) {
    error_log('ERRO: ' . $e->getMessage());
    $errors[] = $e->getMessage();
}

// Passa o número do WhatsApp para o JavaScript
$whatsappNumber = $whatsapp ?? '';
error_log('Número final para WhatsApp: ' . $whatsappNumber);
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo formatPageTitle('Carrinho'); ?></title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/png" href="../favicon.png">
    <link rel="shortcut icon" type="image/png" href="../favicon.png">
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Bootstrap Icons -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.1/font/bootstrap-icons.css" rel="stylesheet">
    <!-- Custom CSS -->
    <link rel="stylesheet" href="../assets/css/theme.css">
    <link rel="stylesheet" href="../assets/css/loja.css">
</head>
<body>
    <!-- Navbar -->
    <nav class="navbar navbar-expand-lg navbar-light bg-white fixed-top">
        <div class="container">
            <!-- Logo -->
            <a class="navbar-brand" href="index.php">
                <?php if (!empty($logo)): ?>
                    <img src="<?php echo $logo; ?>" alt="Logo" class="img-fluid">
                <?php else: ?>
                    <span class="h4 mb-0"><?php echo htmlspecialchars($config_empresa['nome'] ?? 'Minha Loja'); ?></span>
                <?php endif; ?>
            </a>
            
            <!-- Toggle Button -->
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            
            <!-- Menu Items -->
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav ms-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="index.php">Voltar à Loja</a>
                    </li>
                    
                    <!-- Carrinho -->
                    <li class="nav-item ms-lg-3">
                        <a class="nav-link cart-button" href="#" id="cartButton">
                            <i class="bi bi-cart3"></i>
                            <span class="cart-count">0</span>
                        </a>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <!-- Content -->
    <main class="container cart-page mt-5">
        <h4 class="mb-4">Meu Carrinho</h4>
        
        <div class="cart-items">
            <!-- Os itens do carrinho serão carregados aqui via JavaScript -->
        </div>

        <div class="cart-summary mt-4">
            <div class="row">
                <div class="col-md-6 offset-md-6">
                    <div class="card">
                        <div class="card-body">
                            <h5 class="card-title">Resumo do Pedido</h5>
                            <div class="d-flex justify-content-between mb-2">
                                <span>Subtotal:</span>
                                <span class="cart-subtotal">R$ 0,00</span>
                            </div>
                            <hr>
                            <div class="d-flex justify-content-between mb-3">
                                <strong>Total:</strong>
                                <strong class="cart-total">R$ 0,00</strong>
                            </div>
                            <button class="btn btn-success w-100" id="finalizarCompra">
                                <i class="bi bi-whatsapp me-2"></i>Finalizar Pedido via WhatsApp
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Toast Container -->
    <div class="position-fixed bottom-0 end-0 p-3" style="z-index: 11">
        <div id="liveToast" class="toast" role="alert" aria-live="assertive" aria-atomic="true">
            <div class="toast-header">
                <strong class="me-auto">Notificação</strong>
                <button type="button" class="btn-close" data-bs-dismiss="toast" aria-label="Close"></button>
            </div>
            <div class="toast-body"></div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Elementos do DOM
            const cartItemsContainer = document.querySelector('.cart-items');
            const cartSubtotal = document.querySelector('.cart-subtotal');
            const cartTotal = document.querySelector('.cart-total');
            const cartCountElement = document.querySelector('.cart-count');
            
            // Função para formatar preço
            function formatPrice(price) {
                return 'R$ ' + price.toFixed(2).replace('.', ',');
            }
            
            // Função para atualizar o total do carrinho
            function updateCartTotal() {
                const cartItems = JSON.parse(localStorage.getItem('cartItems') || '[]');
                const total = cartItems.reduce((sum, item) => sum + (item.price * item.quantity), 0);
                
                if (cartSubtotal) cartSubtotal.textContent = formatPrice(total);
                if (cartTotal) cartTotal.textContent = formatPrice(total);
            }
            
            // Função para atualizar o contador do carrinho
            function updateCartCount() {
                const cartItems = JSON.parse(localStorage.getItem('cartItems') || '[]');
                const totalItems = cartItems.reduce((sum, item) => sum + item.quantity, 0);
                
                if (cartCountElement) {
                    cartCountElement.textContent = totalItems;
                    cartCountElement.style.display = totalItems > 0 ? 'inline-block' : 'none';
                }
            }
            
            // Função para renderizar os itens do carrinho
            function renderCartItems() {
                const cartItems = JSON.parse(localStorage.getItem('cartItems') || '[]');
                
                if (cartItems.length === 0) {
                    cartItemsContainer.innerHTML = `
                        <div class="text-center py-5">
                            <i class="bi bi-cart3 display-1 text-muted mb-4"></i>
                            <h5 class="text-muted mb-4">Seu carrinho está vazio</h5>
                            <a href="index.php" class="btn btn-primary">
                                <i class="bi bi-arrow-left me-2"></i>Continuar Comprando
                            </a>
                        </div>
                    `;
                    return;
                }
                
                const itemsHtml = cartItems.map(item => `
                    <div class="card mb-3" data-product-id="${item.id}">
                        <div class="card-body">
                            <div class="row align-items-center">
                                <div class="col-md-2">
                                    <img src="${item.image.startsWith('http') ? item.image : '../' + item.image}" class="img-fluid rounded" alt="${item.name}">
                                </div>
                                <div class="col-md-4">
                                    <h5 class="card-title mb-2">${item.name}</h5>
                                    <p class="card-text text-muted mb-0">${item.category}</p>
                                </div>
                                <div class="col-md-2">
                                    <div class="text-muted mb-1">Preço:</div>
                                    <strong>${formatPrice(item.price)}</strong>
                                </div>
                                <div class="col-md-2">
                                    <div class="text-muted mb-2">Quantidade:</div>
                                    <div class="input-group input-group-sm">
                                        <button class="btn btn-outline-secondary decrease-quantity" type="button">
                                            <i class="bi bi-dash"></i>
                                        </button>
                                        <input type="number" class="form-control text-center item-quantity" 
                                               value="${item.quantity}" min="1">
                                        <button class="btn btn-outline-secondary increase-quantity" type="button">
                                            <i class="bi bi-plus"></i>
                                        </button>
                                    </div>
                                </div>
                                <div class="col-md-1">
                                    <div class="text-muted mb-1">Total:</div>
                                    <strong>${formatPrice(item.price * item.quantity)}</strong>
                                </div>
                                <div class="col-md-1 text-end">
                                    <button class="btn btn-link remove-item" title="Remover item">
                                        <i class="bi bi-trash"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                `).join('');
                
                cartItemsContainer.innerHTML = itemsHtml;
                
                // Adiciona eventos aos botões
                cartItemsContainer.querySelectorAll('.decrease-quantity').forEach(btn => {
                    btn.addEventListener('click', function() {
                        const card = this.closest('.card');
                        const productId = card.dataset.productId;
                        const input = card.querySelector('.item-quantity');
                        if (input.value > 1) {
                            input.value = parseInt(input.value) - 1;
                            updateItemQuantity(productId, parseInt(input.value));
                        }
                    });
                });
                
                cartItemsContainer.querySelectorAll('.increase-quantity').forEach(btn => {
                    btn.addEventListener('click', function() {
                        const card = this.closest('.card');
                        const productId = card.dataset.productId;
                        const input = card.querySelector('.item-quantity');
                        input.value = parseInt(input.value) + 1;
                        updateItemQuantity(productId, parseInt(input.value));
                    });
                });
                
                cartItemsContainer.querySelectorAll('.remove-item').forEach(btn => {
                    btn.addEventListener('click', function() {
                        if (confirm('Tem certeza que deseja remover este item?')) {
                            const card = this.closest('.card');
                            const productId = card.dataset.productId;
                            removeItem(productId);
                        }
                    });
                });
                
                cartItemsContainer.querySelectorAll('.item-quantity').forEach(input => {
                    input.addEventListener('change', function() {
                        const card = this.closest('.card');
                        const productId = card.dataset.productId;
                        const quantity = parseInt(this.value);
                        if (quantity < 1) {
                            this.value = 1;
                            updateItemQuantity(productId, 1);
                        } else {
                            updateItemQuantity(productId, quantity);
                        }
                    });
                });
            }
            
            // Função para atualizar quantidade de um item
            function updateItemQuantity(productId, quantity) {
                const cartItems = JSON.parse(localStorage.getItem('cartItems') || '[]');
                const itemIndex = cartItems.findIndex(item => item.id === productId);
                
                if (itemIndex !== -1) {
                    cartItems[itemIndex].quantity = quantity;
                    localStorage.setItem('cartItems', JSON.stringify(cartItems));
                    renderCartItems();
                    updateCartTotal();
                    updateCartCount();
                }
            }
            
            // Função para remover um item
            function removeItem(productId) {
                const cartItems = JSON.parse(localStorage.getItem('cartItems') || '[]');
                const updatedItems = cartItems.filter(item => item.id !== productId);
                localStorage.setItem('cartItems', JSON.stringify(updatedItems));
                renderCartItems();
                updateCartTotal();
                updateCartCount();
                
                // Mostra toast de confirmação
                const toast = new bootstrap.Toast(document.getElementById('liveToast'));
                const toastBody = document.querySelector('.toast-body');
                toastBody.textContent = 'Item removido do carrinho!';
                toast.show();
            }
            
            // Inicializa a página
            renderCartItems();
            updateCartTotal();
            updateCartCount();
            
            // Evento para finalizar compra
            document.getElementById('finalizarCompra').addEventListener('click', function() {
                const cartItems = JSON.parse(localStorage.getItem('cartItems') || '[]');
                
                if (cartItems.length === 0) {
                    alert('Seu carrinho está vazio!');
                    return;
                }
                
                // Formata a mensagem para o WhatsApp
                let message = `🛍️ *NOVO PEDIDO* 🛍️\n\n`;
                message += `📝 *Lista de Produtos:*\n`;
                
                // Lista cada produto
                cartItems.forEach((item, index) => {
                    message += `\n${index + 1}. *${item.name}*\n`;
                    message += `   • Quantidade: ${item.quantity}\n`;
                    message += `   • Valor unitário: ${formatPrice(item.price)}\n`;
                    message += `   • Subtotal: ${formatPrice(item.price * item.quantity)}\n`;
                });
                
                // Calcula e adiciona o total
                const total = cartItems.reduce((sum, item) => sum + (item.price * item.quantity), 0);
                message += `\n💰 *Total do Pedido: ${formatPrice(total)}*\n`;
                
                // Adiciona mensagem final
                message += `\n✨ Olá! Gostaria de fazer este pedido.`;
                
                // Codifica a mensagem para URL
                const encodedMessage = encodeURIComponent(message);
                
                // Pega o número do WhatsApp
                const whatsappNumber = '<?php echo $whatsappNumber; ?>';
                
                if (!whatsappNumber) {
                    alert('Número do WhatsApp não configurado! Por favor, configure o número de celular nas configurações da empresa.');
                    return;
                }
                
                // Abre o WhatsApp com a mensagem
                window.open(`https://api.whatsapp.com/send?phone=55${whatsappNumber}&text=${encodedMessage}`);
            });
        });
    </script>
</body>
</html>
