<?php
// Funções utilitárias

/**
 * Formata um valor para o formato de moeda brasileiro
 */
function formatMoney($value) {
    return number_format($value, 2, ',', '.');
}

/**
 * Converte um valor de moeda brasileiro para float
 */
function unformatMoney($value) {
    return (float) str_replace(['R$', '.', ','], ['', '', '.'], $value);
}

/**
 * Gera um nome único para arquivo
 */
function generateUniqueFilename($originalName) {
    $extension = pathinfo($originalName, PATHINFO_EXTENSION);
    return uniqid() . '_' . time() . '.' . $extension;
}

/**
 * Valida uma imagem
 */
function validateImage($file) {
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
    $maxSize = 5 * 1024 * 1024; // 5MB
    
    if (!in_array($file['type'], $allowedTypes)) {
        throw new Exception('Tipo de arquivo não permitido. Apenas JPG, PNG e GIF são aceitos.');
    }
    
    if ($file['size'] > $maxSize) {
        throw new Exception('Arquivo muito grande. Tamanho máximo permitido: 5MB');
    }
    
    return true;
}

/**
 * Upload de arquivo
 */
function uploadFile($file, $directory) {
    if (!file_exists($directory)) {
        mkdir($directory, 0777, true);
    }
    
    $filename = generateUniqueFilename($file['name']);
    $destination = $directory . '/' . $filename;
    
    if (!move_uploaded_file($file['tmp_name'], $destination)) {
        throw new Exception('Erro ao fazer upload do arquivo');
    }
    
    return $filename;
}

/**
 * Remove arquivo
 */
function removeFile($filename, $directory) {
    $file = $directory . '/' . $filename;
    if (file_exists($file)) {
        unlink($file);
    }
}

/**
 * Sanitiza uma string
 */
function sanitize($string) {
    return htmlspecialchars(strip_tags($string), ENT_QUOTES, 'UTF-8');
}

/**
 * Verifica se é uma requisição POST
 */
function isPost() {
    return $_SERVER['REQUEST_METHOD'] === 'POST';
}

/**
 * Verifica se é uma requisição AJAX
 */
function isAjax() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
}

/**
 * Redireciona para uma URL
 */
function redirect($url) {
    header("Location: $url");
    exit;
}

/**
 * Retorna resposta JSON
 */
function jsonResponse($data) {
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

// Função para buscar as configurações do sistema
function getConfiguracoes() {
    try {
        $pdo = getConnection();
        $stmt = $pdo->query("SELECT * FROM configuracoes LIMIT 1");
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erro ao buscar configurações: " . $e->getMessage());
        return null;
    }
}

/**
 * Função para buscar as configurações do site
 */
function getSiteConfig() {
    try {
        $pdo = getConnection();
        
        // Verifica se a tabela existe
        $stmt = $pdo->query("SHOW TABLES LIKE 'config_site'");
        if ($stmt->rowCount() == 0) {
            return [
                'site_ativo' => true,
                'slide_ativo' => true,
                'slide_texto_principal' => 'Bem-vinda à Loja',
                'slide_texto_secundario' => 'Encontre as melhores peças aqui',
                'nome_empresa' => 'Minha Loja'
            ];
        }
        
        // Busca as configurações
        $stmt = $pdo->query("SELECT * FROM config_site LIMIT 1");
        $config = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$config) {
            // Se não encontrou configurações, insere valores padrão
            $stmt = $pdo->prepare("INSERT INTO config_site (site_ativo, slide_ativo, slide_texto_principal, slide_texto_secundario, nome_empresa) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([
                true,
                true,
                'Bem-vinda à Loja',
                'Encontre as melhores peças aqui',
                'Minha Loja'
            ]);
            
            return [
                'site_ativo' => true,
                'slide_ativo' => true,
                'slide_texto_principal' => 'Bem-vinda à Loja',
                'slide_texto_secundario' => 'Encontre as melhores peças aqui',
                'nome_empresa' => 'Minha Loja'
            ];
        }
        
        return $config;
    } catch (Exception $e) {
        error_log("Erro ao buscar configurações do site: " . $e->getMessage());
        return [
            'site_ativo' => true,
            'slide_ativo' => true,
            'slide_texto_principal' => 'Bem-vinda à Loja',
            'slide_texto_secundario' => 'Encontre as melhores peças aqui',
            'nome_empresa' => 'Minha Loja'
        ];
    }
}

/**
 * Formata o título da página com o nome da empresa
 * @param string $pageTitle Título da página
 * @return string Título formatado
 */
function formatPageTitle($pageTitle) {
    try {
        $pdo = getConnection();
        $stmt = $pdo->query("SELECT nome_empresa FROM config_empresa LIMIT 1");
        $config = $stmt->fetch(PDO::FETCH_ASSOC);
        $empresa = !empty($config['nome_empresa']) ? $config['nome_empresa'] : '';
        
        // Formata o nome da empresa (primeira letra de cada palavra em maiúsculo)
        if (!empty($empresa)) {
            $empresa = ucwords(mb_strtolower($empresa, 'UTF-8'));
            return $pageTitle . ' - ' . $empresa;
        }
        
        return $pageTitle;
    } catch (Exception $e) {
        error_log("Erro ao buscar nome da empresa: " . $e->getMessage());
        return $pageTitle;
    }
}
