$(document).ready(function() {
    let currentPage = 1;
    const itemsPerPage = 10;
    let vendaIdParaCancelar = null;
    let modalImpressao = null;
    let modalCancelarVenda = null;
    let modalPagamentoParcial = null;

    // Inicializa os modais
    modalImpressao = new bootstrap.Modal('#modalImpressao');
    modalCancelarVenda = new bootstrap.Modal('#modalCancelarVenda');
    modalPagamentoParcial = new bootstrap.Modal('#modalPagamentoParcial');

    // Inicializa a máscara para valores monetários
    $('.money').mask('000.000.000,00', {reverse: true});

    // Carrega vendas
    function carregarVendas(page = 1) {
        console.log('Carregando vendas - Página:', page);
        
        const filtros = {
            id: $('#filtroId').val(),
            cliente: $('#filtroCliente').val(),
            data: $('#filtroData').val(),
            valor: $('#filtroValor').val(),
            metodo: $('#filtroMetodo').val()
        };
        
        console.log('Filtros:', filtros);

        $.get('actions/get_vendas.php', { 
            page: page,
            limit: itemsPerPage,
            ...filtros
        })
        .done(function(response) {
            console.log('Resposta recebida:', response);
            
            if (response.success) {
                const vendas = response.data.vendas;
                const totalPages = response.data.pages;
                
                console.log('Total de vendas:', vendas.length);
                console.log('Total de páginas:', totalPages);
                
                // Limpa a tabela
                $('#vendasTableBody').empty();
                
                if (vendas.length === 0) {
                    $('#vendasTableBody').append(`
                        <tr>
                            <td colspan="7" class="text-center">Nenhuma venda encontrada</td>
                        </tr>
                    `);
                    return;
                }
                
                // Adiciona as vendas
                vendas.forEach(function(venda) {
                    // Ignora vendas excluídas
                    if (venda.deleted_at) {
                        console.log('Venda excluída ignorada:', venda.id);
                        return;
                    }

                    const data = new Date(venda.created_at).toLocaleString('pt-BR');
                    const valor = new Intl.NumberFormat('pt-BR', {
                        style: 'currency',
                        currency: 'BRL'
                    }).format(venda.valor_total);
                    
                    const metodoPagamento = formatarMetodoPagamento(venda.metodo_pagamento);
                    let statusBadge = '';
                    let valorRestanteFormatado = '-';
                    
                    if (venda.metodo_pagamento === 'Crediário') {
                        const totalPago = parseFloat(venda.total_pago) || 0;
                        const valorTotal = parseFloat(venda.valor_total);
                        const valorRestante = valorTotal - totalPago;
                        
                        valorRestanteFormatado = new Intl.NumberFormat('pt-BR', {
                            style: 'currency',
                            currency: 'BRL'
                        }).format(valorRestante);

                        if (venda.status === 'Quitado') {
                            statusBadge = '<span class="badge bg-success ms-2">QUITADO</span>';
                            valorRestanteFormatado = 'R$ 0,00';
                        } else {
                            statusBadge = '<span class="badge bg-warning ms-2">PENDENTE</span>';
                        }
                    }
                    
                    let botoesAcao = `
                        <button class="btn btn-sm btn-outline-primary" onclick="imprimirVenda(${venda.id})" title="Imprimir">
                            <i class="bi bi-printer"></i>
                        </button>
                    `;

                    // Se for crediário e não estiver quitado, mostra botão de pagamento parcial
                    if (venda.metodo_pagamento === 'Crediário' && venda.status === 'Pendente') {
                        const valorRestante = venda.valor_total - venda.total_pago;
                        botoesAcao += `
                            <button class="btn btn-sm btn-outline-success ms-1 pagar-parcial" 
                                    data-id="${venda.id}" 
                                    data-valor-restante="${valorRestante}"
                                    data-valor-total="${venda.valor_total}"
                                    title="Pagamento Parcial">
                                <i class="bi bi-cash-coin"></i>
                            </button>
                        `;
                    }

                    // Adiciona botão de cancelar para todas as vendas
                    botoesAcao += `
                        <button class="btn btn-sm btn-outline-danger ms-1" onclick="cancelarVenda(${venda.id})" title="Cancelar">
                            <i class="bi bi-x-circle"></i>
                        </button>
                    `;
                    
                    // Adiciona a linha na tabela
                    $('#vendasTableBody').append(`
                        <tr>
                            <td>${venda.id}</td>
                            <td>${venda.cliente_nome || 'Cliente não informado'}</td>
                            <td>${data}</td>
                            <td>${metodoPagamento}${statusBadge}</td>
                            <td>${valor}</td>
                            <td>${valorRestanteFormatado}</td>
                            <td>${botoesAcao}</td>
                        </tr>
                    `);
                });
                
                // Atualiza a paginação
                atualizarPaginacao(page, totalPages);
            } else {
                console.error('Erro na resposta:', response.message);
                showToast('Erro ao carregar vendas: ' + response.message, 'error');
            }
        })
        .fail(function(jqXHR, textStatus, errorThrown) {
            console.error('Erro na requisição:', {
                status: jqXHR.status,
                textStatus: textStatus,
                errorThrown: errorThrown,
                responseText: jqXHR.responseText
            });
            showToast('Erro ao carregar vendas: ' + textStatus, 'error');
        });
    }

    // Evento de clique no botão de pagamento parcial
    $(document).on('click', '.pagar-parcial', function(e) {
        e.preventDefault();
        const vendaId = $(this).data('id');
        const valorRestante = parseFloat($(this).data('valor-restante'));
        const valorTotal = parseFloat($(this).data('valor-total'));
        
        // Atualiza os valores no modal
        $('#vendaId').val(vendaId);
        $('#valorTotalVenda').text(new Intl.NumberFormat('pt-BR', {
            style: 'currency',
            currency: 'BRL'
        }).format(valorTotal));
        $('#valorRestante').text(new Intl.NumberFormat('pt-BR', {
            style: 'currency',
            currency: 'BRL'
        }).format(valorRestante));

        // Carrega o histórico de pagamentos
        $.get('actions/get_historico_pagamentos.php', { venda_id: vendaId })
            .done(function(response) {
                if (response.success) {
                    $('#historicoPagamentos').empty();
                    response.data.forEach(function(pagamento) {
                        // Usa a data já formatada do PHP
                        const valor = new Intl.NumberFormat('pt-BR', {
                            style: 'currency',
                            currency: 'BRL'
                        }).format(pagamento.valor);
                        
                        $('#historicoPagamentos').append(`
                            <tr>
                                <td>${pagamento.data_pagamento}</td>
                                <td>${valor}</td>
                            </tr>
                        `);
                    });
                }
            });
        
        modalPagamentoParcial.show();
    });

    // Handler do formulário de pagamento parcial
    $('#formPagamentoParcial').submit(function(e) {
        e.preventDefault();
        
        // Pega o valor do input e converte para número
        const valorInput = $('#valorPagamento').val();
        const valor = parseFloat(valorInput.replace(/\./g, '').replace(',', '.'));
        
        console.log('Enviando pagamento:', {
            venda_id: $('#vendaId').val(),
            valor: valor,
            valorOriginal: valorInput
        });

        const formData = new FormData();
        formData.append('venda_id', $('#vendaId').val());
        formData.append('valor', valor);
        
        $.ajax({
            url: 'actions/registrar_pagamento_parcial.php',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                console.log('Resposta do servidor:', response);
                if (response.success) {
                    showToast('Pagamento registrado com sucesso!', 'success');
                    modalPagamentoParcial.hide();
                    carregarVendas(currentPage);
                    $('#formPagamentoParcial').trigger('reset');
                } else {
                    let mensagem = response.error || 'Erro ao registrar pagamento';
                    if (response.debug) {
                        console.error('Detalhes do erro:', response.debug);
                    }
                    showToast(mensagem, 'error');
                }
            },
            error: function(xhr) {
                console.error('Erro na requisição:', xhr);
                let mensagem = 'Erro ao registrar pagamento';
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.error) {
                        mensagem = response.error;
                        if (response.debug) {
                            console.error('Detalhes do erro:', response.debug);
                        }
                    }
                } catch (e) {
                    console.error('Erro ao parsear resposta:', e);
                }
                showToast(mensagem, 'error');
            }
        });
    });

    // Função para formatar método de pagamento
    function formatarMetodoPagamento(metodo) {
        switch (metodo) {
            case 'Dinheiro':
                return '<i class="bi bi-cash me-1"></i> Dinheiro';
            case 'Cartão de Crédito':
                return '<i class="bi bi-credit-card me-1"></i> Cartão de Crédito';
            case 'Cartão de Débito':
                return '<i class="bi bi-credit-card me-1"></i> Cartão de Débito';
            case 'PIX':
                return '<i class="bi bi-phone me-1"></i> PIX';
            case 'Crediário':
                return '<i class="bi bi-journal-text me-1"></i> Crediário';
            default:
                return metodo;
        }
    }

    // Atualiza paginação
    function atualizarPaginacao(currentPage, totalPages) {
        const paginationElement = $('#vendaPagination');
        paginationElement.empty();
        
        if (totalPages <= 1) return;
        
        let html = '<ul class="pagination justify-content-center">';
        
        // Botão Previous
        html += `
            <li class="page-item ${currentPage === 1 ? 'disabled' : ''}">
                <a class="page-link" href="#" data-page="${currentPage - 1}" aria-label="Previous">
                    <span aria-hidden="true">&laquo;</span>
                </a>
            </li>
        `;
        
        // Páginas
        for (let i = 1; i <= totalPages; i++) {
            if (
                i === 1 || // Primeira página
                i === totalPages || // Última página
                (i >= currentPage - 1 && i <= currentPage + 1) // Páginas próximas à atual
            ) {
                html += `
                    <li class="page-item ${i === currentPage ? 'active' : ''}">
                        <a class="page-link" href="#" data-page="${i}">${i}</a>
                    </li>
                `;
            } else if (
                i === currentPage - 2 ||
                i === currentPage + 2
            ) {
                html += '<li class="page-item disabled"><span class="page-link">...</span></li>';
            }
        }
        
        // Botão Next
        html += `
            <li class="page-item ${currentPage === totalPages ? 'disabled' : ''}">
                <a class="page-link" href="#" data-page="${currentPage + 1}" aria-label="Next">
                    <span aria-hidden="true">&raquo;</span>
                </a>
            </li>
        `;
        
        html += '</ul>';
        paginationElement.html(html);
    }

    // Carrega vendas ao iniciar
    $(document).ready(function() {
        console.log('Página carregada, iniciando carregamento de vendas...');
        carregarVendas();
    });

    // Evento de filtro
    $('#btnFiltrar').click(function(e) {
        e.preventDefault();
        console.log('Botão filtrar clicado');
        carregarVendas(1);
    });

    // Evento de limpar filtro
    $('#btnLimparFiltro').click(function(e) {
        e.preventDefault();
        console.log('Botão limpar filtro clicado');
        $('#filtroId').val('');
        $('#filtroCliente').val('');
        $('#filtroData').val('');
        $('#filtroValor').val('');
        $('#filtroMetodo').val('');
        carregarVendas(1);
    });

    // Evento de paginação
    $(document).on('click', '.page-link', function(e) {
        e.preventDefault();
        const page = $(this).data('page');
        console.log('Mudando para página:', page);
        carregarVendas(page);
    });

    // Função para mostrar toast
    function showToast(message, type = 'info') {
        const toast = $(`
            <div class="toast align-items-center text-white bg-${type} border-0" role="alert" aria-live="assertive" aria-atomic="true">
                <div class="d-flex">
                    <div class="toast-body">
                        ${message}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
                </div>
            </div>
        `);
        
        $('.toast-container').append(toast);
        
        const bsToast = new bootstrap.Toast(toast, {
            autohide: true,
            delay: 3000
        });
        
        bsToast.show();
        
        toast.on('hidden.bs.toast', function() {
            $(this).remove();
        });
    }

    // Cancela uma venda
    window.cancelarVenda = function(id) {
        vendaIdParaCancelar = id;
        modalCancelarVenda.show();
    };

    // Confirma o cancelamento da venda
    $('#btnConfirmarCancelamento').click(function() {
        if (vendaIdParaCancelar) {
            $.post('actions/cancelar_venda.php', { id: vendaIdParaCancelar })
                .done(function(response) {
                    if (response.success) {
                        showToast('Venda cancelada com sucesso!', 'success');
                        modalCancelarVenda.hide();
                        carregarVendas(currentPage);
                    } else {
                        showToast('Erro ao cancelar venda: ' + response.message, 'error');
                    }
                })
                .fail(function() {
                    showToast('Erro ao cancelar venda', 'error');
                });
        }
    });

    // Imprime uma venda
    window.imprimirVenda = function(id) {
        // Primeiro busca as configurações da empresa
        $.get('actions/get_config_empresa.php')
            .done(function(configResponse) {
                console.log('Resposta das configurações:', configResponse);
                
                if (!configResponse.success) {
                    showToast('Erro ao buscar configurações: ' + (configResponse.message || 'Erro desconhecido'), 'error');
                    return;
                }

                // Verifica se config existe e tem as propriedades necessárias
                const config = configResponse.config || {};
                const nomeEmpresa = config.nome_empresa || 'MINHA LOJA';
                const enderecoEmpresa = config.endereco_empresa || '';
                const telefoneEmpresa = config.telefone_empresa || '';
                const cnpjEmpresa = config.cnpj_empresa || '';
                
                // Depois busca os detalhes da venda
                $.get('actions/get_venda_detalhes.php', { id: id })
                    .done(function(response) {
                        console.log('Resposta dos detalhes da venda:', response);
                        
                        if (response.success) {
                            const venda = response.data;
                            const data = new Date(venda.created_at).toLocaleString('pt-BR');
                            
                            // Monta o cupom
                            let cupomHtml = `
                                <div class="text-center">
                                    <div class="divider">=============================</div>
                                    <div class="empresa">${nomeEmpresa}</div>
                                    <div class="endereco">${enderecoEmpresa}</div>
                                    <div class="telefone">${telefoneEmpresa}</div>
                                    <div class="cnpj">${cnpjEmpresa}</div>
                                    <div class="divider">=============================</div>
                                    
                                    <div class="info">
                                        <div>Venda Nº: ${venda.id}</div>
                                        <div>Data: ${data}</div>
                                        <div>Cliente: ${venda.cliente_nome || 'Não informado'}</div>
                                    </div>
                                    
                                    <div class="divider">------------------------------</div>
                                    <div class="titulo-secao">ITENS</div>
                                    <div class="divider">------------------------------</div>
                                </div>
                            `;
                            
                            venda.itens.forEach(function(item) {
                                const valorTotal = (item.quantidade * item.valor_unitario).toLocaleString('pt-BR', {
                                    style: 'currency',
                                    currency: 'BRL'
                                });
                                
                                cupomHtml += `
                                    <div class="item text-center">
                                        <div class="produto">${item.produto_nome}</div>
                                        <div class="quantidade">${item.quantidade}x ${item.tamanho}</div>
                                        <div class="valor">${valorTotal}</div>
                                        <div class="divider">------------------------------</div>
                                    </div>
                                `;
                            });
                            
                            cupomHtml += `
                                <div class="text-center">
                                    <div class="divider">=============================</div>
                                    <div class="total">
                                        TOTAL: ${parseFloat(venda.valor_total).toLocaleString('pt-BR', {
                                            style: 'currency',
                                            currency: 'BRL'
                                        })}
                                    </div>
                                    <div class="divider">=============================</div>
                                    
                                    <div class="footer">
                                        <div>${config.cupom_message || 'Obrigado pela preferência! Volte sempre!'}</div>
                                    </div>
                                </div>
                            `;
                            
                            // Exibe o modal com o cupom
                            $('#cupomContent').html(`
                                <style>
                                    /* Estilos específicos para o modal de impressão */
                                    #modalImpressao .modal-dialog {
                                        max-width: 400px;
                                        margin: 1.75rem auto;
                                    }

                                    #modalImpressao .modal-content {
                                        border-radius: 0;
                                        min-height: 500px;
                                    }

                                    #modalImpressao .modal-body {
                                        padding: 20px;
                                    }

                                    #cupomContent {
                                        font-family: 'Courier New', monospace;
                                        font-size: 12px;
                                        line-height: 1.3;
                                        width: 100%;
                                        padding: 15px;
                                        background-color: white;
                                    }

                                    /* Estilos para impressão */
                                    @media print {
                                        body * {
                                            visibility: hidden;
                                        }
                                        #cupomContent, #cupomContent * {
                                            visibility: visible;
                                        }
                                        #cupomContent {
                                            position: absolute;
                                            left: 0;
                                            top: 0;
                                            width: 80mm;
                                            padding: 3mm;
                                        }
                                    }

                                    .text-center {
                                        text-align: center;
                                    }
                                    
                                    .divider {
                                        margin: 3px 0;
                                    }
                                    
                                    .empresa {
                                        font-size: 16px;
                                        font-weight: bold;
                                        margin: 5px 0;
                                    }

                                    .endereco, .telefone, .cnpj {
                                        font-size: 12px;
                                        margin: 3px 0;
                                    }
                                    
                                    .info {
                                        margin: 10px 0;
                                        line-height: 1.4;
                                    }
                                    
                                    .titulo-secao {
                                        font-weight: bold;
                                        margin: 5px 0;
                                        font-size: 14px;
                                    }
                                    
                                    .item {
                                        margin: 5px 0;
                                        text-align: center;
                                    }
                                    
                                    .produto {
                                        font-weight: bold;
                                        margin-bottom: 3px;
                                        font-size: 13px;
                                    }
                                    
                                    .quantidade {
                                        margin: 3px 0;
                                    }
                                    
                                    .valor {
                                        text-align: center;
                                        margin: 3px 0;
                                        font-weight: bold;
                                    }
                                    
                                    .total {
                                        font-size: 14px;
                                        font-weight: bold;
                                        margin: 5px 0;
                                    }
                                    
                                    .footer {
                                        margin-top: 15px;
                                        font-size: 12px;
                                        line-height: 1.4;
                                    }
                                </style>
                                ${cupomHtml}
                            `);
                            modalImpressao.show();
                        } else {
                            showToast('Erro ao buscar detalhes da venda: ' + response.message, 'error');
                        }
                    })
                    .fail(function() {
                        showToast('Erro ao buscar detalhes da venda', 'error');
                    });
            })
            .fail(function() {
                showToast('Erro ao buscar configurações da empresa', 'error');
            });
    };

    // Evento de impressão
    $('#btnImprimirCupom').click(function() {
        const conteudo = $('#cupomContent').html();
        const janelaImpressao = window.open('', '_blank', 'width=800,height=600');
        janelaImpressao.document.write(`
            <!DOCTYPE html>
            <html>
                <head>
                    <meta charset="UTF-8">
                    <title>Cupom de Venda</title>
                    <style>
                        * {
                            margin: 0;
                            padding: 0;
                            box-sizing: border-box;
                        }
                        
                        @page {
                            size: 80mm auto;
                            margin: 0;
                        }
                        
                        html, body {
                            width: 80mm;
                            font-family: 'Courier New', monospace;
                            font-size: 12px;
                            line-height: 1.2;
                            background: white;
                        }
                        
                        #cupom {
                            width: 80mm;
                            padding: 5mm;
                            background: white;
                        }
                        
                        .text-center {
                            text-align: center;
                        }
                        
                        .divider {
                            margin: 2px 0;
                        }
                        
                        .empresa {
                            font-size: 14px;
                            font-weight: bold;
                            margin: 4px 0;
                        }
                        
                        .info {
                            margin: 8px 0;
                            line-height: 1.3;
                        }
                        
                        .titulo-secao {
                            font-weight: bold;
                            margin: 4px 0;
                        }
                        
                        .item {
                            margin: 4px 0;
                            text-align: center;
                        }
                        
                        .produto {
                            font-weight: bold;
                            margin-bottom: 2px;
                        }
                        
                        .quantidade {
                            margin: 2px 0;
                        }
                        
                        .valor {
                            text-align: center;
                            margin: 2px 0;
                            font-weight: bold;
                        }
                        
                        .total {
                            font-size: 12px;
                            font-weight: bold;
                            margin: 4px 0;
                        }
                        
                        .footer {
                            margin-top: 10px;
                            font-size: 10px;
                            line-height: 1.3;
                        }
                        
                        @media screen {
                            body {
                                background: #f0f0f0;
                                display: flex;
                                justify-content: center;
                                align-items: flex-start;
                                min-height: 100vh;
                                padding: 20px;
                            }
                            #cupom {
                                box-shadow: 0 0 10px rgba(0,0,0,0.1);
                            }
                        }
                        
                        @media print {
                            html, body {
                                width: 80mm !important;
                            }
                            #cupom {
                                padding: 0;
                                box-shadow: none;
                            }
                            @page {
                                size: 80mm auto;
                                margin: 0;
                            }
                        }
                    </style>
                </head>
                <body>
                    <div id="cupom">
                        ${conteudo}
                    </div>
                    <script>
                        window.onload = function() {
                            setTimeout(function() {
                                window.print();
                                setTimeout(function() {
                                    window.close();
                                }, 500);
                            }, 250);
                        };
                    </script>
                </body>
            </html>
        `);
        janelaImpressao.document.close();
    });

    // Evento do botão Filtrar
    $('#btnFiltrar').click(function() {
        carregarVendas(1);
    });

    // Evento do botão Limpar Filtro
    $('#btnLimparFiltro').click(function() {
        $('#filtroVendas')[0].reset();
        carregarVendas(1);
    });

    // Evento de submit do formulário (previne submit ao pressionar Enter)
    $('#filtroVendas').on('submit', function(e) {
        e.preventDefault();
        carregarVendas(1);
    });
});
