$(document).ready(function() {
    // Aplicando máscaras aos campos
    $('#cnpj').mask('00.000.000/0000-00');
    $('#cep').mask('00000-000');
    $('#celular').mask('(00) 00000-0000');
    $('#numero').mask('0000000');

    // Validação de email
    function validateEmail(email) {
        var re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }

    // Função para mostrar toast
    function showToast(message, type = 'success') {
        const bgColor = type === 'success' ? 'bg-success' : 'bg-danger';
        const toast = `
            <div class="toast ${bgColor} text-white" role="alert" aria-live="assertive" aria-atomic="true">
                <div class="d-flex">
                    <div class="toast-body">
                        ${message}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
                </div>
            </div>
        `;
        
        $('.toast-container').append(toast);
        const toastElement = $('.toast').last();
        const bsToast = new bootstrap.Toast(toastElement);
        bsToast.show();
        
        // Remove o toast depois que ele é fechado
        toastElement.on('hidden.bs.toast', function () {
            $(this).remove();
        });
    }

    // Preview da logo
    $('#logoInput').change(function(e) {
        if (this.files && this.files[0]) {
            const reader = new FileReader();
            reader.onload = function(e) {
                $('#logoPreview').attr('src', e.target.result);
            };
            reader.readAsDataURL(this.files[0]);
        }
    });

    // Auto completar cidade pelo CEP
    $('#cep').on('blur', function() {
        const cep = $(this).val().replace(/\D/g, '');
        if (cep.length === 8) {
            $.get(`https://viacep.com.br/ws/${cep}/json/`, function(data) {
                if (!data.erro) {
                    $('#cidade').val(data.localidade);
                    $('#bairro').val(data.bairro);
                    $('#endereco').val(data.logradouro);
                }
            });
        }
    });

    // Submissão do formulário da empresa
    $('#empresaForm').on('submit', function(e) {
        e.preventDefault();
        
        // Validações
        const email = $('#email').val();
        if (email && !validateEmail(email)) {
            showToast('Por favor, insira um email válido', 'danger');
            return;
        }

        const cnpj = $('#cnpj').val();
        if (cnpj && cnpj.length < 18) {
            showToast('Por favor, insira um CNPJ válido', 'danger');
            return;
        }

        const cep = $('#cep').val();
        if (cep && cep.length < 9) {
            showToast('Por favor, insira um CEP válido', 'danger');
            return;
        }

        const celular = $('#celular').val();
        if (celular && celular.length < 15) {
            showToast('Por favor, insira um número de celular válido', 'danger');
            return;
        }

        // Criar FormData para enviar arquivos
        const formData = new FormData();
        formData.append('form_type', 'empresa');
        formData.append('nome_empresa', $('#nomeEmpresa').val());
        formData.append('cnpj', $('#cnpj').val());
        formData.append('endereco', $('#endereco').val());
        formData.append('numero', $('#numero').val());
        formData.append('bairro', $('#bairro').val());
        formData.append('cep', $('#cep').val());
        formData.append('cidade', $('#cidade').val());
        formData.append('celular', $('#celular').val());
        formData.append('email', $('#email').val());

        // Adicionar logo se houver
        const logoInput = $('#logoInput')[0];
        if (logoInput.files.length > 0) {
            formData.append('logo', logoInput.files[0]);
        }

        // Enviar dados via AJAX
        $.ajax({
            url: 'actions/process_config.php',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            dataType: 'json',
            success: function(response) {
                console.log('Resposta do servidor:', response);
                if (response && response.success) {
                    showToast(response.message);
                    // Atualizar preview da logo se foi enviada
                    if (logoInput.files.length > 0) {
                        const reader = new FileReader();
                        reader.onload = function(e) {
                            $('#logoPreview').attr('src', e.target.result);
                        };
                        reader.readAsDataURL(logoInput.files[0]);
                    }
                } else {
                    showToast(response.message || 'Erro ao salvar as configurações', 'danger');
                }
            },
            error: function(xhr, status, error) {
                console.error('Erro AJAX:', {
                    status: status,
                    error: error,
                    response: xhr.responseText
                });
                try {
                    const response = JSON.parse(xhr.responseText);
                    showToast(response.message || 'Erro ao salvar as configurações', 'danger');
                } catch (e) {
                    showToast('Erro ao salvar as configurações. Por favor, tente novamente.', 'danger');
                }
            }
        });
    });

    // Submissão do formulário de redes sociais
    $('#redesSociaisForm').on('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData();
        formData.append('form_type', 'redes');
        formData.append('facebook', $('#facebook').val());
        formData.append('instagram', $('#instagram').val());
        formData.append('tiktok', $('#tiktok').val());
        formData.append('linkedin', $('#linkedin').val());

        // Enviar dados via AJAX
        $.ajax({
            url: 'actions/process_config.php',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            dataType: 'json',
            success: function(response) {
                console.log('Resposta do servidor:', response);
                if (response && response.success) {
                    showToast(response.message);
                    // Manter a aba de redes sociais ativa
                    $('#redes-sociais-tab').tab('show');
                } else {
                    showToast(response.message || 'Erro ao salvar as redes sociais', 'danger');
                }
            },
            error: function(xhr, status, error) {
                console.error('Erro AJAX:', {
                    status: status,
                    error: error,
                    response: xhr.responseText
                });
                try {
                    const response = JSON.parse(xhr.responseText);
                    showToast(response.message || 'Erro ao salvar as redes sociais', 'danger');
                } catch (e) {
                    showToast('Erro ao salvar as redes sociais. Por favor, tente novamente.', 'danger');
                }
            }
        });
    });

    // Submissão do formulário de estilo
    $('#estiloForm').on('submit', function(e) {
        e.preventDefault();
        
        // Pegar os valores dos inputs de cor
        const corPrimaria = $('#corPrimaria').val() || $('#corPrimariaHex').val();
        const corSecundaria = $('#corSecundaria').val() || $('#corSecundariaHex').val();
        const corTerciaria = $('#corTerciaria').val() || $('#corTerciariaHex').val();
        const corTexto = $('#corTexto').val() || $('#corTextoHex').val();
        
        console.log('Enviando cores:', {
            corPrimaria,
            corSecundaria,
            corTerciaria,
            corTexto
        });

        const formData = new FormData();
        formData.append('form_type', 'estilo');
        formData.append('cor_primaria', corPrimaria);
        formData.append('cor_secundaria', corSecundaria);
        formData.append('cor_terciaria', corTerciaria);
        formData.append('cor_texto', corTexto);

        // Enviar dados via AJAX
        $.ajax({
            url: 'actions/process_config.php',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            dataType: 'json',
            success: function(response) {
                console.log('Resposta do servidor:', response);
                if (response && response.success) {
                    // Atualizar o tema após salvar as cores
                    $.get('actions/update_theme.php', function(themeResponse) {
                        console.log('Resposta do update_theme:', themeResponse);
                        if (themeResponse.success) {
                            // Recarregar o CSS do tema
                            const themeLink = $('link[href*="theme.css"]');
                            if (themeLink.length) {
                                const newHref = themeLink.attr('href').split('?')[0] + '?v=' + new Date().getTime();
                                themeLink.attr('href', newHref);
                            } else {
                                $('head').append('<link rel="stylesheet" href="assets/css/theme.css?v=' + new Date().getTime() + '">');
                            }

                            // Recarregar os valores dos inputs
                            loadConfigs();
                        }
                    });
                    
                    showToast(response.message);
                } else {
                    showToast(response.message || 'Erro ao salvar o estilo', 'danger');
                }
            },
            error: function(xhr, status, error) {
                console.error('Erro AJAX:', {
                    status: status,
                    error: error,
                    response: xhr.responseText
                });
                try {
                    const response = JSON.parse(xhr.responseText);
                    showToast(response.message || 'Erro ao salvar o estilo', 'danger');
                } catch (e) {
                    showToast('Erro ao salvar o estilo. Por favor, tente novamente.', 'danger');
                }
            }
        });
    });

    // Submissão do formulário do cupom
    $('#cupomForm').on('submit', function(e) {
        e.preventDefault();
        
        // Criar FormData com todos os campos necessários
        const formData = new FormData();
        formData.append('form_type', 'empresa');
        
        // Campos obrigatórios da empresa (mantém os valores existentes)
        formData.append('nome_empresa', $('#nomeEmpresa').val() || '');
        formData.append('cnpj', $('#cnpj').val() || '');
        formData.append('telefone', $('#telefone').val() || '');
        formData.append('whatsapp', $('#whatsapp').val() || '');
        formData.append('email', $('#email').val() || '');
        formData.append('instagram', $('#instagram').val() || '');
        formData.append('endereco', $('#endereco').val() || '');
        formData.append('numero', $('#numero').val() || '');
        formData.append('bairro', $('#bairro').val() || '');
        formData.append('cep', $('#cep').val() || '');
        formData.append('cidade', $('#cidade').val() || '');
        formData.append('celular', $('#celular').val() || '');
        
        // Campo do cupom que está sendo alterado
        formData.append('cupom_message', $('#cupomMessage').val());
        
        // Enviar dados
        $.ajax({
            url: 'actions/save_config_empresa.php',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success) {
                    showToast('Configurações do cupom salvas com sucesso!');
                } else {
                    showToast(response.message || 'Erro ao salvar configurações', 'danger');
                }
            },
            error: function(xhr, status, error) {
                showToast('Erro ao carregar configurações: ' + error, 'danger');
            }
        });
    });

    // Função para sincronizar colorpicker com campo hexadecimal
    function setupColorSync(colorPickerId, hexInputId) {
        const colorPicker = $(colorPickerId);
        const hexInput = $(hexInputId);

        // Quando o colorpicker muda
        colorPicker.on('input', function() {
            hexInput.val(this.value.toUpperCase());
            updateThemePreview();
        });

        // Quando o campo hexadecimal muda
        hexInput.on('input', function() {
            let value = this.value;
            // Adiciona # se não existir
            if (!value.startsWith('#')) {
                value = '#' + value;
            }
            // Remove caracteres inválidos
            value = value.replace(/[^#0-9A-Fa-f]/g, '');
            // Limita a 7 caracteres (#RRGGBB)
            value = value.substring(0, 7);
            // Atualiza o campo
            this.value = value.toUpperCase();
            
            // Se for um valor hexadecimal válido
            if (/^#[0-9A-Fa-f]{6}$/.test(value)) {
                colorPicker.val(value);
                updateThemePreview();
            }
        });

        // Quando o campo hexadecimal perde o foco
        hexInput.on('blur', function() {
            let value = this.value;
            // Se não for um valor hexadecimal válido, restaura o valor do colorpicker
            if (!/^#[0-9A-Fa-f]{6}$/.test(value)) {
                this.value = colorPicker.val().toUpperCase();
            }
        });
    }

    // Configurar sincronização para cada par de campos
    setupColorSync('#corPrimaria', '#corPrimariaHex');
    setupColorSync('#corSecundaria', '#corSecundariaHex');
    setupColorSync('#corTerciaria', '#corTerciariaHex');
    setupColorSync('#corTexto', '#corTextoHex');

    // Função para atualizar o preview do tema
    function updateThemePreview() {
        const root = document.documentElement;
        const colors = {
            '--cor-principal': $('#corPrimaria').val(),
            '--cor-secundaria': $('#corSecundaria').val(),
            '--cor-terciaria': $('#corTerciaria').val(),
            '--cor-texto': $('#corTexto').val()
        };

        // Atualizar variáveis CSS
        Object.entries(colors).forEach(([property, value]) => {
            root.style.setProperty(property, value);
            // Converter hex para RGB
            const rgb = hexToRgb(value);
            if (rgb) {
                root.style.setProperty(property + '-rgb', `${rgb.r}, ${rgb.g}, ${rgb.b}`);
            }
        });
    }

    // Função para converter hex para RGB
    function hexToRgb(hex) {
        const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
        return result ? {
            r: parseInt(result[1], 16),
            g: parseInt(result[2], 16),
            b: parseInt(result[3], 16)
        } : null;
    }

    // Carregar configurações iniciais
    function loadConfigs() {
        $.get('actions/get_config_empresa.php')
            .done(function(response) {
                if (response.success) {
                    const config = response.config;
                    $('#nomeEmpresa').val(config.nome_empresa);
                    $('#cnpj').val(config.cnpj);
                    $('#endereco').val(config.endereco);
                    $('#numero').val(config.numero);
                    $('#bairro').val(config.bairro);
                    $('#cep').val(config.cep);
                    $('#cidade').val(config.cidade);
                    $('#celular').val(config.celular);
                    $('#email').val(config.email);
                    $('#instagram').val(config.instagram);
                    $('#cupomMessage').val(config.cupom_message);
                }
            })
            .fail(function(xhr, status, error) {
                showToast('Erro ao carregar configurações: ' + error, 'danger');
            });

        $.get('actions/get_config.php?type=redes', function(response) {
            if (response.success && response.data) {
                const data = response.data;
                $('#facebook').val(data.facebook || '');
                $('#instagram').val(data.instagram || '');
                $('#tiktok').val(data.tiktok || '');
                $('#linkedin').val(data.linkedin || '');
            }
        });

        $.get('actions/get_config.php?type=estilo', function(response) {
            if (response.success && response.data) {
                const data = response.data;
                $('#corPrimaria').val(data.cor_principal || '#1923ae');
                $('#corPrimariaHex').val(data.cor_principal || '#1923ae');
                $('#corSecundaria').val(data.cor_secundaria || '#b88700');
                $('#corSecundariaHex').val(data.cor_secundaria || '#b88700');
                $('#corTerciaria').val(data.cor_terciaria || '#166053');
                $('#corTerciariaHex').val(data.cor_terciaria || '#166053');
                $('#corTexto').val(data.cor_texto || '#166053');
                $('#corTextoHex').val(data.cor_texto || '#166053');
            }
        });
    }

    // Carregar status do site
    function carregarStatusSite() {
        $.ajax({
            url: 'actions/get_config_site.php',
            method: 'GET',
            dataType: 'json'
        })
        .done(function(response) {
            if (response.success) {
                $('#ativar_site').prop('checked', response.site_ativo);
                $('#status_text').text(response.site_ativo ? 'Site Ativo' : 'Site Desativado');
            } else {
                console.error('Erro:', response.error);
                showToast('Erro ao carregar status do site', 'danger');
            }
        })
        .fail(function(xhr) {
            console.error('Erro ao carregar status do site:', xhr.responseText);
            showToast('Erro ao carregar status do site', 'danger');
        });
    }

    // Atualizar texto do status quando mudar o switch
    $('#ativar_site').change(function() {
        const status = $(this).is(':checked');
        $('#status_text').text(status ? 'Site Ativo' : 'Site Desativado');
    });

    // Salvar configurações do site
    $('#salvar_config_site').click(function() {
        const status = $('#ativar_site').is(':checked');
        
        $.ajax({
            url: 'actions/atualizar_status_site.php',
            method: 'POST',
            dataType: 'json',
            data: { status: status }
        })
        .done(function(response) {
            if (response.success) {
                showToast('Configurações do site salvas com sucesso!');
            } else {
                showToast(response.error || 'Erro ao salvar configurações', 'danger');
                carregarStatusSite(); // Recarrega o status atual
            }
        })
        .fail(function(xhr) {
            console.error('Erro ao salvar:', xhr.responseText);
            showToast('Erro ao salvar configurações do site', 'danger');
            carregarStatusSite(); // Recarrega o status atual
        });
    });

    // Carregar status do site ao iniciar
    carregarStatusSite();

    // Carregar configurações ao iniciar
    loadConfigs();

    // Manter a aba ativa após submissão
    $('.nav-link').on('click', function (e) {
        e.preventDefault();
        $(this).tab('show');
    });

    // Restaurar última aba ativa
    var lastTab = localStorage.getItem('lastTab');
    if (lastTab) {
        $(`#${lastTab}`).tab('show');
    }

    // Salvar aba ativa quando mudar
    $('.nav-link').on('shown.bs.tab', function (e) {
        localStorage.setItem('lastTab', $(e.target).attr('id'));
    });

    // Importação do banco de dados
    $('#formImportDatabase').on('submit', function(e) {
        e.preventDefault();
        
        if (!confirm('ATENÇÃO: Importar um banco de dados irá substituir todos os dados atuais. Tem certeza que deseja continuar?')) {
            return;
        }

        const formData = new FormData(this);
        
        $.ajax({
            url: 'actions/import_database.php',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            beforeSend: function() {
                showLoading();
            },
            success: function(response) {
                hideLoading();
                const data = JSON.parse(response);
                if (data.success) {
                    showToast('Banco de dados importado com sucesso!');
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                } else {
                    showToast(data.error || 'Erro ao importar banco de dados', 'danger');
                }
            },
            error: function(xhr) {
                hideLoading();
                const response = JSON.parse(xhr.responseText);
                showToast(response.error || 'Erro ao importar banco de dados', 'danger');
            }
        });
    });
});
