<?php
// Habilita exibição de erros
error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json');

require_once '../includes/config.php';
require_once '../includes/functions.php';

// Log dos dados recebidos
error_log("=== INÍCIO DO PROCESSAMENTO DE PAGAMENTO PARCIAL ===");
error_log("POST recebido: " . print_r($_POST, true));

// Verifica se é uma requisição POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    error_log("Método inválido: " . $_SERVER['REQUEST_METHOD']);
    http_response_code(405);
    echo json_encode(['error' => 'Método não permitido']);
    exit;
}

// Obtém e valida os dados
$venda_id = filter_input(INPUT_POST, 'venda_id', FILTER_VALIDATE_INT);
$valor = filter_input(INPUT_POST, 'valor', FILTER_VALIDATE_FLOAT);

error_log("Dados filtrados - venda_id: " . var_export($venda_id, true));
error_log("Dados filtrados - valor: " . var_export($valor, true));

if (!$venda_id || !$valor) {
    $erro = 'Dados inválidos - venda_id: ' . var_export($venda_id, true) . ', valor: ' . var_export($valor, true);
    error_log("Erro de validação: " . $erro);
    http_response_code(400);
    echo json_encode([
        'error' => $erro,
        'debug' => [
            'venda_id' => $venda_id,
            'valor' => $valor,
            'post' => $_POST
        ]
    ]);
    exit;
}

try {
    $pdo = getConnection();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    error_log("Conexão com banco estabelecida");
    
    $pdo->beginTransaction();
    error_log("Transação iniciada");

    // Busca informações da venda
    $query = "
        SELECT o.*, 
        COALESCE((
            SELECT SUM(valor)
            FROM pagamentos_parciais
            WHERE venda_id = o.id
        ), 0) as total_pago
        FROM operacoes o
        WHERE o.id = ? AND o.tipo = 'venda'
    ";
    
    error_log("Query de busca da venda: " . $query);
    error_log("Parâmetros da busca - venda_id: " . $venda_id);
    
    $stmt = $pdo->prepare($query);
    $stmt->execute([$venda_id]);
    $venda = $stmt->fetch(PDO::FETCH_ASSOC);

    error_log("Resultado da busca da venda: " . print_r($venda, true));

    if (!$venda) {
        throw new Exception('Venda não encontrada');
    }

    if (strtolower($venda['metodo_pagamento']) !== 'crediario') {
        throw new Exception('Esta venda não é um crediário');
    }

    $valor_restante = $venda['valor_total'] - $venda['total_pago'];
    error_log("Valor restante calculado: " . $valor_restante);
    error_log("Valor do pagamento: " . $valor);

    // Verifica se o valor do pagamento é válido
    if ($valor <= 0) {
        throw new Exception('Valor do pagamento deve ser maior que zero');
    }

    if ($valor > $valor_restante) {
        throw new Exception("Valor do pagamento (R$ $valor) maior que o valor restante (R$ $valor_restante)");
    }

    // Registra o pagamento parcial
    $query = "
        INSERT INTO pagamentos_parciais (venda_id, valor, data_pagamento)
        VALUES (?, ?, NOW())
    ";
    
    error_log("Query de inserção do pagamento: " . $query);
    
    $stmt = $pdo->prepare($query);
    $stmt->execute([$venda_id, $valor]);
    $pagamento_id = $pdo->lastInsertId();
    
    error_log("Pagamento registrado com ID: " . $pagamento_id);

    // Registra na tabela pagamentos_crediario para afetar o caixa
    $stmt = $pdo->prepare("
        INSERT INTO pagamentos_crediario (
            venda_id,
            valor_pago,
            data_pagamento,
            usuario_id,
            cliente_id
        ) VALUES (?, ?, NOW(), ?, ?)
    ");
    $stmt->execute([
        $venda_id,
        $valor,
        $_SESSION['usuario_id'] ?? null,
        $venda['cliente_id']
    ]);

    // Atualiza o status da venda se foi quitada
    $novo_total_pago = $venda['total_pago'] + $valor;
    error_log("Novo total pago: " . $novo_total_pago);
    
    if ($novo_total_pago >= $venda['valor_total']) {
        $query = "
            UPDATE operacoes
            SET status = 'Quitado'
            WHERE id = ?
        ";
        
        error_log("Query de atualização de status: " . $query);
        error_log("Parâmetro: venda_id = $venda_id");
        
        $stmt = $pdo->prepare($query);
        $stmt->execute([$venda_id]);
        error_log("Status da venda atualizado para Quitado");
    }

    // Registra a entrada no caixa
    $query = "
        INSERT INTO movimentacoes_caixa (tipo, valor, data_hora, descricao)
        VALUES ('entrada', ?, NOW(), ?)
    ";
    
    error_log("Query de registro no caixa: " . $query);
    error_log("Parâmetros: valor = $valor, descrição = Pagamento parcial da venda #$venda_id");
    
    $stmt = $pdo->prepare($query);
    $stmt->execute([
        $valor,
        "Pagamento parcial da venda #$venda_id"
    ]);
    error_log("Movimento de caixa registrado com sucesso");

    $pdo->commit();
    error_log("Transação commitada com sucesso");

    $response = [
        'success' => true,
        'message' => 'Pagamento registrado com sucesso',
        'novo_total_pago' => $novo_total_pago,
        'valor_restante' => $venda['valor_total'] - $novo_total_pago,
        'quitado' => $novo_total_pago >= $venda['valor_total']
    ];
    
    error_log("Resposta de sucesso: " . print_r($response, true));
    echo json_encode($response);

} catch (Exception $e) {
    error_log("ERRO NO PROCESSAMENTO: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
        error_log("Transação revertida");
    }
    
    http_response_code(500);
    echo json_encode([
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ]
    ]);
}

error_log("=== FIM DO PROCESSAMENTO DE PAGAMENTO PARCIAL ===");
