<?php
header('Content-Type: application/json');
require_once '../config/database.php';

// Habilitar exibição de erros para debug
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Função para log
function logError($message, $data = null) {
    $log = date('Y-m-d H:i:s') . " - " . $message . "\n";
    if ($data) {
        $log .= print_r($data, true) . "\n";
    }
    error_log($log, 3, "../logs/error.log");
}

// Função para gerar favicon
function generateFavicon($logo_path) {
    if (!extension_loaded('gd')) {
        logError("A extensão GD do PHP não está instalada.");
        return false;
    }

    if (!file_exists($logo_path)) {
        logError("Arquivo da logo não encontrado: " . $logo_path);
        return false;
    }

    // Detecta o tipo de imagem
    $image_info = getimagesize($logo_path);
    if ($image_info === false) {
        logError("Arquivo não é uma imagem válida");
        return false;
    }

    // Carrega a imagem baseado no tipo
    $source = null;
    switch ($image_info[2]) {
        case IMAGETYPE_JPEG:
            $source = imagecreatefromjpeg($logo_path);
            break;
        case IMAGETYPE_PNG:
            $source = imagecreatefrompng($logo_path);
            break;
        default:
            logError("Tipo de imagem não suportado. Use JPG ou PNG.");
            return false;
    }

    if ($source) {
        // Cria uma nova imagem 32x32
        $favicon = imagecreatetruecolor(32, 32);
        // Define fundo transparente
        imagealphablending($favicon, false);
        imagesavealpha($favicon, true);
        $transparent = imagecolorallocatealpha($favicon, 255, 255, 255, 127);
        imagefilledrectangle($favicon, 0, 0, 32, 32, $transparent);
        // Redimensiona
        imagecopyresampled($favicon, $source, 0, 0, 0, 0, 32, 32, imagesx($source), imagesy($source));
        // Salva como PNG
        $favicon_path = __DIR__ . '/../favicon.png';
        $result = imagepng($favicon, $favicon_path);
        // Libera a memória
        imagedestroy($source);
        imagedestroy($favicon);
        
        return $result;
    }
    
    return false;
}

try {
    $database = new Database();
    $db = $database->getConnection();

    $form_type = $_POST['form_type'] ?? '';
    logError("Recebendo requisição do tipo: " . $form_type, $_POST);

    $response = ['success' => 0, 'message' => 'Tipo de formulário não especificado'];

    // Função para salvar histórico
    function saveConfigHistory($db, $type, $data, $description = '') {
        $query = "INSERT INTO config_history (
            config_type, 
            config_data, 
            description, 
            ip_address,
            revert_hash
        ) VALUES (
            :type,
            :data,
            :description,
            :ip,
            :hash
        )";

        $stmt = $db->prepare($query);
        $ip = $_SERVER['REMOTE_ADDR'];
        $hash = md5($type . json_encode($data) . time());

        return $stmt->execute([
            ':type' => $type,
            ':data' => json_encode($data),
            ':description' => $description,
            ':ip' => $ip,
            ':hash' => $hash
        ]);
    }

    switch($form_type) {
        case 'empresa':
            $query = "UPDATE config_empresa SET 
                      nome_empresa = :nome_empresa,
                      cnpj = :cnpj,
                      endereco = :endereco,
                      numero = :numero,
                      bairro = :bairro,
                      cep = :cep,
                      cidade = :cidade,
                      celular = :celular,
                      email = :email
                      WHERE id = 1";

            $stmt = $db->prepare($query);

            $data = [
                'nome_empresa' => $_POST['nome_empresa'],
                'cnpj' => $_POST['cnpj'],
                'endereco' => $_POST['endereco'],
                'numero' => $_POST['numero'],
                'bairro' => $_POST['bairro'],
                'cep' => $_POST['cep'],
                'cidade' => $_POST['cidade'],
                'celular' => $_POST['celular'],
                'email' => $_POST['email']
            ];

            $stmt->bindParam(':nome_empresa', $data['nome_empresa']);
            $stmt->bindParam(':cnpj', $data['cnpj']);
            $stmt->bindParam(':endereco', $data['endereco']);
            $stmt->bindParam(':numero', $data['numero']);
            $stmt->bindParam(':bairro', $data['bairro']);
            $stmt->bindParam(':cep', $data['cep']);
            $stmt->bindParam(':cidade', $data['cidade']);
            $stmt->bindParam(':celular', $data['celular']);
            $stmt->bindParam(':email', $data['email']);

            if ($stmt->execute()) {
                // Salvar histórico
                saveConfigHistory($db, 'empresa', $data, 'Atualização das configurações da empresa');

                // Processar upload do logo se houver
                if (isset($_FILES['logo']) && $_FILES['logo']['error'] === 0) {
                    $upload_dir = '../uploads/';
                    if (!is_dir($upload_dir)) {
                        mkdir($upload_dir, 0777, true);
                    }

                    $file_extension = strtolower(pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION));
                    $new_filename = 'logo.' . $file_extension;
                    $upload_path = $upload_dir . $new_filename;

                    if (move_uploaded_file($_FILES['logo']['tmp_name'], $upload_path)) {
                        // Atualiza o caminho do logo no banco
                        $logo_path = 'uploads/' . $new_filename;
                        $query = "UPDATE config_empresa SET logo = :logo WHERE id = 1";
                        $stmt = $db->prepare($query);
                        $stmt->bindParam(':logo', $logo_path);
                        
                        if ($stmt->execute()) {
                            logError("Logo atualizado com sucesso: " . $logo_path);
                            
                            // Gera o favicon automaticamente
                            if (generateFavicon($upload_path)) {
                                $response = ['success' => 1, 'message' => 'Configurações salvas e favicon gerado com sucesso!'];
                            } else {
                                $response = ['success' => 1, 'message' => 'Configurações salvas, mas houve um erro ao gerar o favicon.'];
                            }
                        }
                    }
                } else {
                    $response = ['success' => 1, 'message' => 'Configurações salvas com sucesso!'];
                }
            }
            break;

        case 'redes':
            $query = "UPDATE config_redes_sociais SET 
                      facebook = :facebook,
                      instagram = :instagram,
                      tiktok = :tiktok,
                      linkedin = :linkedin
                      WHERE id = 1";

            $stmt = $db->prepare($query);

            $data = [
                'facebook' => $_POST['facebook'],
                'instagram' => $_POST['instagram'],
                'tiktok' => $_POST['tiktok'],
                'linkedin' => $_POST['linkedin']
            ];

            $stmt->bindParam(':facebook', $data['facebook']);
            $stmt->bindParam(':instagram', $data['instagram']);
            $stmt->bindParam(':tiktok', $data['tiktok']);
            $stmt->bindParam(':linkedin', $data['linkedin']);

            if ($stmt->execute()) {
                // Salvar histórico
                saveConfigHistory($db, 'redes_sociais', $data, 'Atualização das redes sociais');
                $response = ['success' => 1, 'message' => 'Configurações de redes sociais salvas com sucesso!'];
            }
            break;

        case 'estilo':
            $query = "UPDATE config_estilo_cores SET 
                      cor_principal = :cor_principal,
                      cor_secundaria = :cor_secundaria,
                      cor_terciaria = :cor_terciaria,
                      cor_texto = :cor_texto
                      WHERE id = 1";

            $stmt = $db->prepare($query);

            $data = [
                'cor_principal' => $_POST['cor_primaria'],
                'cor_secundaria' => $_POST['cor_secundaria'],
                'cor_terciaria' => $_POST['cor_terciaria'],
                'cor_texto' => $_POST['cor_texto']
            ];

            $stmt->bindParam(':cor_principal', $data['cor_principal']);
            $stmt->bindParam(':cor_secundaria', $data['cor_secundaria']);
            $stmt->bindParam(':cor_terciaria', $data['cor_terciaria']);
            $stmt->bindParam(':cor_texto', $data['cor_texto']);

            if ($stmt->execute()) {
                // Salvar histórico
                saveConfigHistory($db, 'estilo', $data, 'Atualização das cores do tema');
                logError("Dados de estilo recebidos:", $_POST);
                
                // Atualizar o arquivo theme.css
                $update_url = 'http://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . '/update_theme.php';
                $ch = curl_init($update_url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                $response = curl_exec($ch);
                curl_close($ch);
                
                $response = ['success' => 1, 'message' => 'Configurações de estilo salvas com sucesso!'];
            } else {
                logError("Erro na execução:", $stmt->errorInfo()[2]);
                throw new Exception($stmt->errorInfo()[2]);
            }
            break;
    }

} catch (Exception $e) {
    logError("Erro na execução:", $e->getMessage());
    $response = ['success' => 0, 'message' => 'Erro: ' . $e->getMessage()];
}

echo json_encode($response);
exit;
