<?php
// Garante que nenhum erro PHP será exibido no JSON
error_reporting(0);
ini_set('display_errors', 0);

// Garante que o conteúdo será JSON
header('Content-Type: application/json; charset=utf-8');

// Função para retornar erro em formato JSON
function returnError($message, $code = 500) {
    http_response_code($code);
    echo json_encode([
        'success' => false,
        'message' => $message
    ]);
    exit;
}

// Função para formatar o método de pagamento
function formatarMetodoPagamento($metodo) {
    if (empty($metodo)) return 'À Vista';
    
    $metodo = strtolower(trim($metodo));
    switch ($metodo) {
        case 'dinheiro': return 'Dinheiro';
        case 'crediario': return 'Crediário';
        case 'credito': return 'Cartão de Crédito';
        case 'debito': return 'Cartão de Débito';
        case 'pix': return 'PIX';
        default: return ucfirst($metodo);
    }
}

try {
    require_once __DIR__ . '/../includes/config.php';
    require_once __DIR__ . '/../includes/functions.php';
    
    $pdo = getConnection();
    
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 10;
    $offset = ($page - 1) * $limit;

    // Log para debug
    error_log("Buscando vendas - Página: $page, Limite: $limit, Offset: $offset");

    // Filtros
    $where = ['o.tipo = "venda"'];
    $params = [];
    
    // Filtro por ID
    if (!empty($_GET['id'])) {
        $where[] = "o.id = ?";
        $params[] = $_GET['id'];
    }
    
    // Filtro por cliente
    if (!empty($_GET['cliente'])) {
        $where[] = "c.nome LIKE ?";
        $params[] = "%" . $_GET['cliente'] . "%";
    }
    
    // Filtro por data
    if (!empty($_GET['data'])) {
        $where[] = "DATE(o.created_at) = ?";
        $params[] = $_GET['data'];
    }
    
    // Filtro por valor
    if (!empty($_GET['valor'])) {
        $where[] = "o.valor_total = ?";
        $params[] = $_GET['valor'];
    }
    
    // Filtro por método de pagamento
    if (!empty($_GET['metodo'])) {
        $where[] = "o.metodo_pagamento = ?";
        $params[] = $_GET['metodo'];
    }
    
    // Monta a query
    $whereClause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";
    
    // Conta total de registros
    $queryCount = "SELECT COUNT(*) as total FROM operacoes o LEFT JOIN clientes c ON o.cliente_id = c.id {$whereClause}";
    $stmtCount = $pdo->prepare($queryCount);
    $stmtCount->execute($params);
    $totalRows = $stmtCount->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Busca os registros
    $query = "
        SELECT 
        o.*, 
        c.nome as cliente_nome,
        COALESCE((
            SELECT SUM(valor)
            FROM pagamentos_parciais
            WHERE venda_id = o.id
        ), 0) as total_pago
        FROM operacoes o 
        LEFT JOIN clientes c ON o.cliente_id = c.id 
        {$whereClause}
        ORDER BY o.created_at DESC
        LIMIT ? OFFSET ?
    ";
    
    // Adiciona limit e offset aos parâmetros
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $vendas = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calcula total de páginas
    $totalPages = ceil($totalRows / $limit);

    // Log para debug
    error_log("Total de registros: $totalRows, Total de páginas: $totalPages");

    // Processar os resultados
    foreach ($vendas as &$venda) {
        // Formata o método de pagamento
        $venda['metodo_pagamento'] = formatarMetodoPagamento($venda['metodo_pagamento']);

        // Calcular se está quitado para vendas no crediário
        if (strtolower($venda['metodo_pagamento']) === 'crediário') {
            $venda['status'] = $venda['total_pago'] >= $venda['valor_total'] ? 'Quitado' : 'Pendente';
        }

        // Debug para cada venda
        error_log("Venda ID: " . $venda['id'] . 
                 " | Método: " . $venda['metodo_pagamento'] . 
                 " | Status: " . $venda['status'] . 
                 " | Total: " . $venda['valor_total'] . 
                 " | Pago: " . $venda['total_pago']);
    }

    $response = [
        'success' => true,
        'data' => [
            'vendas' => $vendas,
            'pages' => $totalPages,
            'current_page' => $page,
            'total_records' => $totalRows
        ]
    ];

    // Log para debug
    error_log("Resposta JSON: " . json_encode($response));

    echo json_encode($response);

} catch (Exception $e) {
    error_log("Erro ao buscar vendas: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    returnError("Erro ao buscar vendas: " . $e->getMessage());
}
