<?php
header('Content-Type: application/json');
require_once '../includes/config.php';
require_once '../includes/functions.php';

try {
    $pdo = getConnection();

    // Processa os filtros
    $filtros = [];
    $params = [];

    if (!empty($_GET['periodo'])) {
        $datas = explode(' - ', $_GET['periodo']);
        $dataInicio = DateTime::createFromFormat('d/m/Y', $datas[0])->format('Y-m-d');
        $dataFim = DateTime::createFromFormat('d/m/Y', $datas[1])->format('Y-m-d');
        $filtros[] = "DATE(o.created_at) BETWEEN ? AND ?";
        $params[] = $dataInicio;
        $params[] = $dataFim;
    }

    if (!empty($_GET['cliente'])) {
        $filtros[] = "o.cliente_id = ?";
        $params[] = $_GET['cliente'];
    }

    if (!empty($_GET['produto'])) {
        $filtros[] = "oi.produto_id = ?";
        $params[] = $_GET['produto'];
    }

    if (!empty($_GET['tipoOperacao'])) {
        $filtros[] = "o.tipo = ?";
        $params[] = $_GET['tipoOperacao'];
    }

    $whereClause = $filtros ? "WHERE " . implode(" AND ", $filtros) : "";

    // Dados para o card de resumo
    $query = "
        SELECT 
            COUNT(*) as quantidade_vendas,
            SUM(valor_total) as total_vendas,
            SUM(valor_total)/COUNT(*) as ticket_medio
        FROM operacoes o
        LEFT JOIN operacao_itens oi ON o.id = oi.operacao_id
        $whereClause
        AND o.tipo = 'venda'
        AND o.deleted_at IS NULL
    ";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $resumo = $stmt->fetch(PDO::FETCH_ASSOC);

    // Taxa de conversão de provas
    $query = "
        SELECT 
            COUNT(CASE WHEN status = 'convertida' THEN 1 END) * 100.0 / COUNT(*) as taxa_conversao
        FROM operacoes
        WHERE tipo = 'prova'
        AND deleted_at IS NULL
    ";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $taxaConversao = $stmt->fetch(PDO::FETCH_ASSOC);

    // Vendas por período (últimos 30 dias)
    $query = "
        SELECT 
            DATE(created_at) as data,
            SUM(valor_total) as total
        FROM operacoes
        WHERE tipo = 'venda'
        AND deleted_at IS NULL
        AND created_at >= DATE_SUB(CURRENT_DATE, INTERVAL 30 DAY)
        GROUP BY DATE(created_at)
        ORDER BY data
    ";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $vendasPorPeriodo = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Produtos mais vendidos
    $query = "
        SELECT 
            p.nome,
            SUM(oi.quantidade) as quantidade
        FROM operacao_itens oi
        JOIN produtos p ON oi.produto_id = p.id
        JOIN operacoes o ON oi.operacao_id = o.id
        WHERE o.tipo = 'venda'
        AND o.deleted_at IS NULL
        GROUP BY p.id, p.nome
        ORDER BY quantidade DESC
        LIMIT 5
    ";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $produtosMaisVendidos = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Vendas por forma de pagamento
    $query = "
        SELECT 
            metodo_pagamento,
            COUNT(*) as quantidade
        FROM operacoes
        WHERE tipo = 'venda'
        AND deleted_at IS NULL
        GROUP BY metodo_pagamento
    ";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $vendasPorPagamento = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Melhores clientes
    $query = "
        SELECT 
            c.nome,
            COUNT(*) as quantidade_compras,
            SUM(o.valor_total) as total_compras
        FROM operacoes o
        JOIN clientes c ON o.cliente_id = c.id
        WHERE o.tipo = 'venda'
        AND o.deleted_at IS NULL
        GROUP BY c.id, c.nome
        ORDER BY total_compras DESC
        LIMIT 5
    ";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $melhoresClientes = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Calcula tendências (comparação com período anterior)
    $tendencias = [
        'vendas' => 15.5,  // Exemplo: aumento de 15.5%
        'quantidade' => 8.2,
        'ticket' => 6.7,
        'conversao' => 12.3
    ];

    // Formata os dados para os gráficos
    $dadosGraficos = [
        'vendasPorPeriodo' => [
            'labels' => array_column($vendasPorPeriodo, 'data'),
            'valores' => array_column($vendasPorPeriodo, 'total')
        ],
        'produtosMaisVendidos' => [
            'labels' => array_column($produtosMaisVendidos, 'nome'),
            'valores' => array_column($produtosMaisVendidos, 'quantidade')
        ],
        'vendasPorPagamento' => [
            'labels' => array_column($vendasPorPagamento, 'metodo_pagamento'),
            'valores' => array_column($vendasPorPagamento, 'quantidade')
        ],
        'melhoresClientes' => [
            'labels' => array_column($melhoresClientes, 'nome'),
            'valores' => array_column($melhoresClientes, 'total_compras')
        ]
    ];

    // Monta a resposta
    $response = [
        'success' => true,
        'data' => [
            'resumo' => [
                'totalVendas' => floatval($resumo['total_vendas']),
                'quantidadeVendas' => intval($resumo['quantidade_vendas']),
                'ticketMedio' => floatval($resumo['ticket_medio']),
                'taxaConversao' => floatval($taxaConversao['taxa_conversao']),
                'tendenciaVendas' => $tendencias['vendas'],
                'tendenciaQuantidade' => $tendencias['quantidade'],
                'tendenciaTicket' => $tendencias['ticket'],
                'tendenciaConversao' => $tendencias['conversao']
            ],
            'graficos' => $dadosGraficos
        ]
    ];

    echo json_encode($response);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao carregar relatórios: ' . $e->getMessage()
    ]);
}
