<?php
require_once '../includes/config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

function enviarMensagemWhatsApp($telefone, $mensagem, $imagemUrl = null) {
    $telefone = preg_replace('/[^0-9]/', '', $telefone);
    if (strlen($telefone) !== 11) {
        throw new Exception("Número de telefone inválido: {$telefone}");
    }

    $data = [
        'phone' => $telefone,
        'message' => $mensagem
    ];

    if ($imagemUrl) {
        $data['imageUrl'] = $imagemUrl;
    }

    error_log("Enviando mensagem para WhatsApp: " . print_r($data, true));

    $ch = curl_init('http://localhost:3001/send');
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    error_log("Resposta do servidor WhatsApp - HTTP Code: {$httpCode}, Response: {$response}, Error: {$error}");

    if ($httpCode !== 200) {
        throw new Exception("Erro ao enviar mensagem via WhatsApp: " . ($response ?: $error));
    }

    return json_decode($response, true);
}

try {
    error_log("Iniciando envio de campanha");
    $pdo = getConnection();
    
    // Validação dos dados recebidos
    if (!isset($_POST['nome']) || empty($_POST['nome'])) {
        throw new Exception('O nome da campanha é obrigatório');
    }
    
    if (!isset($_POST['descricao']) || empty($_POST['descricao'])) {
        throw new Exception('A descrição da campanha é obrigatória');
    }
    
    if (!isset($_POST['clientes']) || empty($_POST['clientes'])) {
        throw new Exception('Selecione pelo menos um cliente');
    }
    
    // Recebe os dados do formulário
    $nome = $_POST['nome'];
    $descricao = $_POST['descricao'];
    $clientes = json_decode($_POST['clientes'], true);
    
    error_log("Dados recebidos - Nome: {$nome}, Descrição: {$descricao}, Clientes: " . print_r($clientes, true));
    
    if (!is_array($clientes) || empty($clientes)) {
        throw new Exception('Lista de clientes inválida');
    }
    
    // Processa a imagem se foi enviada
    $imagem = null;
    $imagemUrl = null;
    if (isset($_FILES['imagem']) && $_FILES['imagem']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = '../uploads/';
        if (!file_exists($uploadDir)) {
            if (!mkdir($uploadDir, 0777, true)) {
                throw new Exception('Erro ao criar diretório de uploads');
            }
        }
        
        // Validação do tipo de arquivo
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
        $fileType = $_FILES['imagem']['type'];
        if (!in_array($fileType, $allowedTypes)) {
            throw new Exception('Tipo de arquivo não permitido. Use apenas JPG, PNG ou GIF.');
        }
        
        // Validação do tamanho (máximo 5MB)
        if ($_FILES['imagem']['size'] > 5 * 1024 * 1024) {
            throw new Exception('A imagem deve ter no máximo 5MB');
        }
        
        $extension = strtolower(pathinfo($_FILES['imagem']['name'], PATHINFO_EXTENSION));
        $imageName = uniqid('marketing_') . '.' . $extension;
        $uploadFile = $uploadDir . $imageName;
        
        if (!move_uploaded_file($_FILES['imagem']['tmp_name'], $uploadFile)) {
            throw new Exception('Erro ao salvar a imagem');
        }
        $imagem = $imageName;
        $imagemUrl = 'http://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . '/../uploads/' . $imageName;
        error_log("Imagem salva: {$imagemUrl}");
    }
    
    // Inicia transação
    $pdo->beginTransaction();
    
    try {
        // Insere a campanha no banco
        $query = "
            INSERT INTO marketing_campanhas (
                nome, 
                descricao, 
                imagem, 
                data_envio,
                status
            ) VALUES (
                :nome,
                :descricao,
                :imagem,
                NOW(),
                'pendente'
            )
        ";
        
        $stmt = $pdo->prepare($query);
        $stmt->execute([
            ':nome' => $nome,
            ':descricao' => $descricao,
            ':imagem' => $imagem
        ]);
        
        $campanhaId = $pdo->lastInsertId();
        error_log("Campanha criada com ID: {$campanhaId}");
        
        // Prepara a mensagem
        $mensagem = "*{$nome}*\n\n{$descricao}";  // Título em negrito
        $sucessos = 0;
        $erros = 0;
        $resultados = [];
        $erroDetalhes = [];
        
        // Inicia o envio das mensagens
        foreach ($clientes as $cliente) {
            try {
                // Busca o telefone do cliente
                $queryTelefone = "SELECT telefone, nome FROM clientes WHERE id = ?";
                $stmtTelefone = $pdo->prepare($queryTelefone);
                $stmtTelefone->execute([$cliente['id']]);
                $clienteData = $stmtTelefone->fetch(PDO::FETCH_ASSOC);
                
                error_log("Dados do cliente: " . print_r($clienteData, true));
                
                if (!$clienteData || !$clienteData['telefone']) {
                    throw new Exception("Telefone não encontrado para o cliente ID " . $cliente['id']);
                }
                
                // Insere o destinatário
                $queryDestinatario = "
                    INSERT INTO marketing_destinatarios (
                        campanha_id,
                        cliente_id,
                        status,
                        created_at
                    ) VALUES (
                        :campanha_id,
                        :cliente_id,
                        'pendente',
                        NOW()
                    )
                ";
                
                $stmt = $pdo->prepare($queryDestinatario);
                $stmt->execute([
                    ':campanha_id' => $campanhaId,
                    ':cliente_id' => $cliente['id']
                ]);
                
                $destinatarioId = $pdo->lastInsertId();
                
                // Envia a mensagem via WhatsApp
                $response = enviarMensagemWhatsApp($clienteData['telefone'], $mensagem, $imagemUrl);
                
                // Atualiza o status do envio
                $queryUpdate = "
                    UPDATE marketing_destinatarios 
                    SET status = 'enviado', 
                        data_envio = NOW() 
                    WHERE id = ?
                ";
                $stmtUpdate = $pdo->prepare($queryUpdate);
                $stmtUpdate->execute([$destinatarioId]);
                
                $sucessos++;
                
            } catch (Exception $e) {
                $erros++;
                $erro = "Erro ao enviar mensagem para cliente {$cliente['id']} ({$clienteData['nome']}): " . $e->getMessage();
                error_log($erro);
                $erroDetalhes[] = $erro;
                
                // Atualiza o status com erro
                if (isset($destinatarioId)) {
                    $queryUpdateError = "
                        UPDATE marketing_destinatarios 
                        SET status = 'erro', 
                            data_envio = NOW() 
                        WHERE id = ?
                    ";
                    $stmtUpdateError = $pdo->prepare($queryUpdateError);
                    $stmtUpdateError->execute([$destinatarioId]);
                }
            }
        }
        
        // Atualiza o status da campanha
        $status = $erros === 0 ? 'enviado' : ($sucessos === 0 ? 'erro' : 'parcial');
        $queryUpdateCampanha = "
            UPDATE marketing_campanhas 
            SET status = ? 
            WHERE id = ?
        ";
        $stmtUpdateCampanha = $pdo->prepare($queryUpdateCampanha);
        $stmtUpdateCampanha->execute([$status, $campanhaId]);
        
        // Confirma a transação
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => "Campanha registrada com sucesso! {$sucessos} mensagens enviadas" . 
                        ($erros > 0 ? ", {$erros} erros" : ""),
            'erros' => $erroDetalhes
        ]);
        
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        throw $e;
    }

} catch (Exception $e) {
    error_log("Erro fatal: " . $e->getMessage());
    
    // Se houver transação em andamento, faz rollback
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    // Se houver imagem salva e ocorrer erro, remove a imagem
    if (isset($uploadFile) && file_exists($uploadFile)) {
        unlink($uploadFile);
    }
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
