<?php
// Habilitar exibição de erros para debug
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once '../includes/config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

// Log do POST recebido
error_log("POST recebido: " . print_r($_POST, true));

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Método não permitido']);
    exit;
}

if (empty($_POST['prova_id'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'ID da prova não fornecido']);
    exit;
}

try {
    $pdo = getConnection();
    $pdo->beginTransaction();

    $provaId = $_POST['prova_id'];
    error_log("Processando prova ID: " . $provaId);

    // Verificar se a prova existe e não foi deletada
    $stmt = $pdo->prepare("
        SELECT o.*, c.nome as cliente_nome 
        FROM operacoes o
        JOIN clientes c ON o.cliente_id = c.id 
        WHERE o.id = ? 
        AND o.tipo = 'prova' 
        AND o.deleted_at IS NULL
    ");
    $stmt->execute([$provaId]);
    $prova = $stmt->fetch();
    
    error_log("Resultado da busca da prova: " . print_r($prova, true));
    
    if (!$prova) {
        throw new Exception('Prova não encontrada ou já foi processada');
    }

    // Buscar itens da prova
    $stmt = $pdo->prepare("
        SELECT oi.*, p.nome as produto_nome 
        FROM operacao_itens oi
        JOIN produtos p ON oi.produto_id = p.id 
        WHERE oi.operacao_id = ? 
        AND oi.deleted_at IS NULL
    ");
    $stmt->execute([$provaId]);
    $itens = $stmt->fetchAll(PDO::FETCH_ASSOC);

    error_log("Itens encontrados: " . print_r($itens, true));

    if (empty($itens)) {
        throw new Exception('Nenhum item encontrado para esta prova');
    }

    // Devolver cada item ao estoque
    foreach ($itens as $item) {
        error_log("Processando item: " . print_r($item, true));
        
        // Primeiro verifica se o produto_tamanho existe
        $stmt = $pdo->prepare("
            SELECT id, estoque as estoque_atual 
            FROM produto_tamanhos 
            WHERE produto_id = ? AND tamanho = ?
        ");
        $stmt->execute([$item['produto_id'], $item['tamanho']]);
        $produtoTamanho = $stmt->fetch();

        error_log("Produto tamanho encontrado: " . print_r($produtoTamanho, true));

        if (!$produtoTamanho) {
            throw new Exception("Tamanho {$item['tamanho']} não encontrado para o produto {$item['produto_nome']}");
        }

        // Atualiza o estoque
        $stmt = $pdo->prepare("
            UPDATE produto_tamanhos 
            SET estoque = estoque + ?
            WHERE produto_id = ? 
            AND tamanho = ?
        ");
        $stmt->execute([
            $item['quantidade'],
            $item['produto_id'],
            $item['tamanho']
        ]);

        if ($stmt->rowCount() === 0) {
            throw new Exception("Erro ao atualizar estoque do produto {$item['produto_nome']} tamanho {$item['tamanho']}");
        }
    }

    // Marcar a prova e seus itens como deletados
    $now = date('Y-m-d H:i:s');
    
    $stmt = $pdo->prepare("
        UPDATE operacoes 
        SET deleted_at = ? 
        WHERE id = ?
    ");
    $stmt->execute([$now, $provaId]);

    if ($stmt->rowCount() === 0) {
        throw new Exception('Erro ao marcar prova como deletada');
    }

    $stmt = $pdo->prepare("
        UPDATE operacao_itens 
        SET deleted_at = ? 
        WHERE operacao_id = ?
    ");
    $stmt->execute([$now, $provaId]);

    $pdo->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Itens devolvidos ao estoque com sucesso'
    ]);

} catch (Exception $e) {
    error_log("Erro na devolução: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    
    if (isset($pdo)) {
        $pdo->rollBack();
    }
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao devolver itens: ' . $e->getMessage()
    ]);
}
