$(document).ready(function() {
    // Inicializa as máscaras
    $('.money').mask('000.000.000,00', {reverse: true});

    // Inicializa DataTables
    const tabelaProdutos = $('#tabelaProdutos').DataTable({
        language: {
            url: 'assets/js/i18n/pt-BR.json'
        },
        ajax: {
            url: 'actions/get_produtos.php',
            dataSrc: ''
        },
        columns: [
            { 
                data: 'imagem',
                title: 'Imagem',
                width: '10%',
                render: function(data, type, row) {
                    return `<img src="uploads/produtos/${data}" class="produto-thumbnail" alt="${row.nome}">`;
                }
            },
            { 
                data: 'nome',
                title: 'Nome',
                width: '20%'
            },
            { 
                data: 'categoria',
                title: 'Categoria',
                width: '15%'
            },
            { 
                data: null,
                title: 'Tamanho e Estoque',
                width: '20%',
                render: function(data, type, row) {
                    if (type === 'display') {
                        let tamanhos = row.tamanhos.join(' - ');
                        let estoques = row.estoques.join(' - ');
                        return `<div class="tamanhos">${tamanhos}</div>
                                <div class="estoques">${estoques}</div>`;
                    }
                    return row.tamanhos.join(',');
                }
            },
            { 
                data: 'estoque_total',
                title: 'Estoque Total',
                width: '15%'
            },
            { 
                data: 'valor',
                title: 'Valor',
                width: '80px',
                render: function(data) {
                    return formatarMoeda(data);
                }
            },
            {
                data: null,
                title: 'Ações',
                width: '10%',
                orderable: false,
                render: function(data, type, row) {
                    return `
                        <button class="btn btn-sm btn-primary editar-produto" data-id="${row.id}" type="button">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-sm btn-danger excluir-produto" data-id="${row.id}" data-nome="${row.nome}" type="button">
                            <i class="bi bi-trash"></i>
                        </button>
                    `;
                }
            }
        ],
        order: [[1, 'asc']]
    });

    let tabelaCategorias;
    
    // Inicializa a tabela de categorias
    function inicializarTabelaCategorias() {
        tabelaCategorias = $('#tabelaCategorias').DataTable({
            dom: '<"row"<"col-12"l>>' +  
                '<"row"<"col-12"f>>' +  
                '<"row"<"col-12"t>>' +  
                '<"row"<"col-12"i>>' +  
                '<"row"<"col-12"p>>',   
            pageLength: 8,
            language: {
                url: 'assets/js/i18n/pt-BR.json',
                search: "Pesquisar:",
                searchPlaceholder: "Buscar registro"
            },
            ajax: {
                url: 'actions/get_categorias.php',
                dataSrc: 'data'
            },
            columns: [
                { data: 'nome' },
                {
                    data: 'id',
                    render: function(data, type, row) {
                        return `
                            <button class="btn btn-sm btn-primary editar-categoria" data-id="${data}" data-nome="${row.nome}">
                                <i class="bi bi-pencil"></i>
                            </button>
                            <button class="btn btn-sm btn-danger excluir-categoria" data-id="${data}" data-nome="${row.nome}">
                                <i class="bi bi-trash"></i>
                            </button>
                        `;
                    }
                }
            ]
        });
    }

    // Inicializa a tabela de categorias
    inicializarTabelaCategorias();

    // Variável para armazenar o ID do produto a ser excluído
    let produtoIdParaExcluir = null;

    // Handler para o botão de excluir
    $(document).on('click', '.excluir-produto', function(e) {
        e.preventDefault();
        produtoIdParaExcluir = $(this).data('id');
        const nomeProduto = $(this).data('nome');
        
        // Atualiza o texto do modal
        $('#modalExcluir .modal-body p').text(`Tem certeza que deseja excluir o produto "${nomeProduto}"?`);
        
        // Mostra o modal usando Bootstrap
        const modalExcluir = document.getElementById('modalExcluir');
        if (modalExcluir) {
            const bsModal = bootstrap.Modal.getOrCreateInstance(modalExcluir);
            bsModal.show();
        }
    });

    // Handler para confirmar exclusão
    $('#confirmar-exclusao').click(function() {
        if (!produtoIdParaExcluir) return;

        $.ajax({
            url: 'actions/delete_produto.php',
            method: 'POST',
            data: { id: produtoIdParaExcluir },
            success: function(response) {
                // Esconde o modal usando Bootstrap
                const modalExcluir = document.getElementById('modalExcluir');
                if (modalExcluir) {
                    const bsModal = bootstrap.Modal.getInstance(modalExcluir);
                    if (bsModal) {
                        bsModal.hide();
                    }
                }
                
                showToast('Produto excluído com sucesso!', 'success');
                tabelaProdutos.ajax.reload();
                produtoIdParaExcluir = null;
            },
            error: function(xhr, status, error) {
                showToast('Erro ao excluir produto: ' + error, 'danger');
            }
        });
    });

    // Variável para armazenar o ID da categoria a ser excluída
    let categoriaIdParaExcluir = null;

    // Handler para o botão de excluir categoria
    $(document).on('click', '.excluir-categoria', function(e) {
        e.preventDefault();
        categoriaIdParaExcluir = $(this).data('id');
        const nomeCategoria = $(this).data('nome');
        
        // Atualiza o texto do modal
        $('#modalExcluirCategoria .modal-body p').text(`Tem certeza que deseja excluir a categoria "${nomeCategoria}"?`);
        
        // Mostra o modal usando Bootstrap
        const modalExcluirCategoria = document.getElementById('modalExcluirCategoria');
        if (modalExcluirCategoria) {
            const bsModal = bootstrap.Modal.getOrCreateInstance(modalExcluirCategoria);
            bsModal.show();
        }
    });

    // Handler para confirmar exclusão de categoria
    $('#confirmar-exclusao-categoria').click(function() {
        if (!categoriaIdParaExcluir) return;

        $.ajax({
            url: 'actions/delete_categoria.php',
            method: 'POST',
            dataType: 'json',
            data: { id: categoriaIdParaExcluir },
            success: function(response) {
                console.log('Resposta da exclusão:', response);
                
                if (response.success) {
                    // Esconde o modal usando Bootstrap
                    const modalExcluirCategoria = document.getElementById('modalExcluirCategoria');
                    if (modalExcluirCategoria) {
                        const bsModal = bootstrap.Modal.getInstance(modalExcluirCategoria);
                        if (bsModal) {
                            bsModal.hide();
                        }
                    }
                    
                    showToast(response.message, 'success');
                    tabelaCategorias.ajax.reload();
                    carregarCategorias();
                    categoriaIdParaExcluir = null;
                } else {
                    showToast(response.message || 'Erro ao excluir categoria', 'error');
                }
            },
            error: function(xhr, status, error) {
                console.error('Erro na requisição:', { xhr, status, error });
                let mensagem = 'Erro ao excluir categoria';
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.message) {
                        mensagem = response.message;
                    }
                } catch (e) {
                    console.error('Erro ao parsear resposta:', e);
                }
                showToast(mensagem, 'danger');
            }
        });
    });

    // Carrega categorias no select
    function carregarCategorias() {
        $.get('actions/get_categorias.php')
            .done(function(response) {
                if (response.success) {
                    const select = $('select[name="categoria_id"]');
                    select.find('option:not(:first)').remove();
                    
                    response.data.forEach(function(categoria) {
                        select.append(`<option value="${categoria.id}">${categoria.nome}</option>`);
                    });
                }
            });
    }

    // Formatar moeda
    function formatarMoeda(valor) {
        return new Intl.NumberFormat('pt-BR', {
            style: 'currency',
            currency: 'BRL'
        }).format(valor);
    }

    // Adicionar novo campo de tamanho
    $('#addTamanho').click(function() {
        const novoTamanho = $('.tamanho-item:first').clone();
        novoTamanho.find('input').val('');
        $('#tamanhos-container').append(novoTamanho);
    });

    // Remover campo de tamanho
    $(document).on('click', '.remover-tamanho', function() {
        if ($('.tamanho-item').length > 1) {
            $(this).closest('.tamanho-item').remove();
        }
    });

    // Salvar produto
    $('#formProduto').submit(function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        
        // Validação dos campos obrigatórios
        const nome = formData.get('nome');
        const valor = formData.get('valor');
        const categoria_id = formData.get('categoria_id');
        const cor = formData.get('cor');
        
        if (!nome || !valor || !categoria_id || !cor) {
            showToast('Por favor, preencha todos os campos obrigatórios', 'error');
            return;
        }
        
        // Validação dos tamanhos
        let temTamanho = false;
        $('.tamanho-item').each(function() {
            const tamanho = $(this).find('[name="tamanhos[]"]').val();
            const estoque = $(this).find('[name="estoques[]"]').val();
            if (tamanho && estoque) {
                temTamanho = true;
            }
        });
        
        if (!temTamanho) {
            showToast('Adicione pelo menos um tamanho e estoque', 'error');
            return;
        }

        // Se não tem imagem e é edição, remove o campo imagem do FormData
        const id = formData.get('id');
        const imagem = formData.get('imagem');
        if (id && (!imagem || imagem.size === 0)) {
            formData.delete('imagem');
        }
        
        $.ajax({
            url: 'actions/save_produto.php',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success) {
                    $('#modalProduto').modal('hide');
                    $('#formProduto')[0].reset();
                    $('#previewImagem').attr('src', '').hide();
                    $('#tamanhos-container').html($('.tamanho-item:first').clone());
                    tabelaProdutos.ajax.reload();
                    showToast(response.message || 'Produto salvo com sucesso!');
                } else {
                    showToast(response.message || 'Erro ao salvar produto', 'error');
                }
            },
            error: function() {
                showToast('Erro ao salvar produto', 'error');
            }
        });
    });

    // Salvar categoria
    $('#formCategoria').submit(function(e) {
        e.preventDefault();
        
        $.ajax({
            url: 'actions/save_categoria.php',
            type: 'POST',
            data: $(this).serialize(),
            success: function(response) {
                if (response.success) {
                    $('#modalCategoria').modal('hide');
                    $('#formCategoria')[0].reset();
                    tabelaCategorias.ajax.reload();
                    carregarCategorias();
                    showToast(response.message || 'Categoria salva com sucesso!');
                } else {
                    showToast(response.message || 'Erro ao salvar categoria', 'error');
                }
            },
            error: function() {
                showToast('Erro ao salvar categoria', 'error');
            }
        });
    });

    // Edição de Categoria
    $(document).on('click', '.editar-categoria', function() {
        const id = $(this).data('id');
        const nome = $(this).data('nome');
        
        $('#formEditarCategoria input[name="id"]').val(id);
        $('#formEditarCategoria input[name="nome"]').val(nome);
        $('#modalEditarCategoria').modal('show');
    });

    $('#formEditarCategoria').submit(function(e) {
        e.preventDefault();
        
        $.post('actions/update_categoria.php', $(this).serialize())
            .done(function(response) {
                if (response.success) {
                    $('#modalEditarCategoria').modal('hide');
                    tabelaCategorias.ajax.reload();
                    carregarCategorias();
                    showToast('Categoria atualizada com sucesso!');
                } else {
                    showToast(response.message || 'Erro ao atualizar categoria', 'error');
                }
            })
            .fail(function() {
                showToast('Erro ao atualizar categoria', 'error');
            });
    });

    // Função para carregar dados do produto para edição
    function carregarProdutoParaEdicao(id) {
        $.get('actions/get_produto.php', { id: id })
            .done(function(response) {
                if (response.success) {
                    const produto = response.data;
                    
                    // Preenche os campos do formulário
                    $('#formProduto')[0].reset(); // Limpa o formulário primeiro
                    $('#formProduto input[name="id"]').val(produto.id);
                    $('#formProduto input[name="nome"]').val(produto.nome);
                    // Formata o valor sem o símbolo R$
                    $('#formProduto input[name="valor"]').val(produto.valor.toString().replace('.', ','));
                    $('#formProduto select[name="categoria_id"]').val(produto.categoria_id);
                    $('#formProduto input[name="cor"]').val(produto.cor);
                    
                    // Limpa e prepara o container de tamanhos
                    $('#tamanhos-container').html('');
                    $('#tamanhos-container').append('<h6 class="mb-3">Tamanhos e Estoque</h6>');
                    
                    // Adiciona os tamanhos do produto
                    if (produto.tamanhos && produto.tamanhos.length > 0) {
                        produto.tamanhos.forEach(function(item) {
                            const novoTamanho = $(`
                                <div class="tamanho-item mb-2">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <input type="text" class="form-control" name="tamanhos[]" placeholder="Tamanho" required value="${item.tamanho}">
                                        </div>
                                        <div class="col-md-5">
                                            <input type="number" class="form-control" name="estoques[]" placeholder="Quantidade" required min="0" value="${item.estoque}">
                                        </div>
                                        <div class="col-md-1">
                                            <button type="button" class="btn btn-danger btn-sm remover-tamanho">
                                                <i class="bi bi-trash"></i>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            `);
                            $('#tamanhos-container').append(novoTamanho);
                        });
                    } else {
                        // Se não tem tamanhos, adiciona um campo vazio
                        const novoTamanho = $(`
                            <div class="tamanho-item mb-2">
                                <div class="row">
                                    <div class="col-md-6">
                                        <input type="text" class="form-control" name="tamanhos[]" placeholder="Tamanho" required>
                                    </div>
                                    <div class="col-md-5">
                                        <input type="number" class="form-control" name="estoques[]" placeholder="Quantidade" required min="0">
                                    </div>
                                    <div class="col-md-1">
                                        <button type="button" class="btn btn-danger btn-sm remover-tamanho">
                                            <i class="bi bi-trash"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        `);
                        $('#tamanhos-container').append(novoTamanho);
                    }
                    
                    // Atualiza preview da imagem se existir
                    if (produto.imagem) {
                        $('#previewImagem').attr('src', 'uploads/produtos/' + produto.imagem).show();
                    } else {
                        $('#previewImagem').attr('src', '').hide();
                    }
                    
                    // Atualiza título do modal
                    $('#modalProdutoLabel').text('Editar Produto');
                    
                    // Abre o modal
                    $('#modalProduto').modal('show');
                } else {
                    showToast(response.message || 'Erro ao carregar produto', 'error');
                }
            })
            .fail(function() {
                showToast('Erro ao carregar produto', 'error');
            });
    }

    // Handler para o botão de editar
    $(document).on('click', '.editar-produto', function() {
        const id = $(this).data('id');
        carregarProdutoParaEdicao(id);
    });

    // Carrega as categorias inicialmente
    carregarCategorias();

    // Preview da imagem
    $('input[name="imagem"]').change(function() {
        const file = this.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = function(e) {
                $('#previewImagem').attr('src', e.target.result).show();
            };
            reader.readAsDataURL(file);
        } else {
            $('#previewImagem').attr('src', '').hide();
        }
    });

    // Limpa os formulários quando os modais são fechados
    $('.modal').on('hidden.bs.modal', function() {
        const $form = $(this).find('form');
        if ($form.length > 0) {
            $form[0].reset();
            
            // Se for o modal de produto
            if ($(this).attr('id') === 'modalProduto') {
                // Limpa o preview da imagem
                $('#previewImagem').attr('src', '').hide();
                
                // Reseta os campos de tamanho
                const primeiroTamanho = $('.tamanho-item:first').clone();
                primeiroTamanho.find('input').val('');
                $('#tamanhos-container').html('');
                $('#tamanhos-container').append('<h6 class="mb-3">Tamanhos e Estoque</h6>');
                $('#tamanhos-container').append(primeiroTamanho);
                
                // Reseta o título do modal
                $('#modalProdutoLabel').text('Cadastrar Produto');
            }
        }
    });
});
