$(document).ready(function() {
    // Verifica o status do caixa antes de qualquer operação
    $.get('actions/check_caixa_status.php')
        .done(function(response) {
            if (!response.success) {
                showToast('Erro ao verificar status do caixa', 'error');
                return;
            }
            
            if (!response.caixaAberto) {
                // Pega a cor primária do CSS
                const primaryColor = getComputedStyle(document.documentElement)
                    .getPropertyValue('--primary-color').trim();
                
                Swal.fire({
                    title: 'Caixa Fechado',
                    text: 'É necessário abrir o caixa antes de realizar vendas.',
                    icon: 'warning',
                    confirmButtonText: 'Ir para o Caixa',
                    showCancelButton: true,
                    cancelButtonText: 'Cancelar',
                    confirmButtonColor: primaryColor,
                    cancelButtonColor: '#6c757d'
                }).then((result) => {
                    if (result.isConfirmed) {
                        window.location.href = 'caixa.php';
                    }
                });
                
                // Desabilita todos os elementos do PDV
                $('#formVenda').find('input, select, button').prop('disabled', true);
                $('#btnFinalizar, #btnCancelar').prop('disabled', true);
                return;
            }
            
            // Se o caixa estiver aberto, inicializa o PDV normalmente
            initializePDV();
        })
        .fail(function() {
            showToast('Erro ao verificar status do caixa', 'error');
        });
});

// Função para inicializar o PDV
function initializePDV() {
    // Inicializa Select2
    $('.select2').select2({
        theme: 'bootstrap-5',
        width: '100%'
    });

    // Carrega clientes
    $.get('actions/get_clientes.php', function(data) {
        const response = typeof data === 'string' ? JSON.parse(data) : data;
        if (response.success && response.data) {
            response.data.forEach(cliente => {
                $('#cliente').append(new Option(cliente.nome, cliente.id));
            });
        }
    });

    // Carrega produtos
    $.get('actions/get_produtos_pdv.php', function(data) {
        const response = typeof data === 'string' ? JSON.parse(data) : data;
        if (response.success && response.data) {
            response.data.forEach(produto => {
                $('#produto').append(new Option(produto.nome, produto.id));
            });
        }
    });

    // Atualiza data e hora
    function atualizarDataHora() {
        const agora = new Date();
        const dataHora = agora.toLocaleString('pt-BR');
        $('#dataHora').text(dataHora);
    }
    atualizarDataHora();
    setInterval(atualizarDataHora, 1000);

    // Variáveis globais
    let carrinho = [];
    let totalGeral = 0;

    // Quando selecionar um produto
    $('#produto').on('change', function() {
        const produtoId = $(this).val();
        if (produtoId) {
            $.get('actions/get_produto_tamanhos_pdv.php', { id: produtoId }, function(data) {
                const response = typeof data === 'string' ? JSON.parse(data) : data;
                if (response.success && response.data) {
                    $('#tamanho').empty().append('<option value="">Selecione um tamanho</option>');
                    response.data.tamanhos.forEach((tamanho, index) => {
                        $('#tamanho').append(new Option(tamanho + ' (Estoque: ' + response.data.estoques[index] + ')', tamanho));
                    });
                    $('#tamanho, #quantidade').prop('disabled', false);
                }
            });
        } else {
            $('#tamanho, #quantidade').prop('disabled', true).val('');
            $('#quantidade').val(1);
        }
    });

    // Quando selecionar um tamanho
    $('#tamanho').on('change', function() {
        const tamanho = $(this).val();
        const tamanhoText = $(this).find('option:selected').text();
        if (tamanho) {
            // Extrai o estoque do texto do option (formato: "TAM (Estoque: X)")
            const estoque = parseInt(tamanhoText.match(/Estoque: (\d+)/)[1]);
            const quantidade = parseInt($('#quantidade').val());
            
            // Só habilita o botão se houver estoque suficiente
            $('#adicionarProduto').prop('disabled', quantidade > estoque);
            
            if (quantidade > estoque) {
                showToast('Quantidade maior que o estoque disponível', 'error');
            }
        } else {
            $('#adicionarProduto').prop('disabled', true);
        }
    });

    // Quando mudar a quantidade
    $('#quantidade').on('change', function() {
        const quantidade = parseInt($(this).val());
        const tamanhoText = $('#tamanho option:selected').text();
        
        if (tamanhoText) {
            const estoque = parseInt(tamanhoText.match(/Estoque: (\d+)/)[1]);
            
            // Verifica se a quantidade é maior que o estoque
            if (quantidade > estoque) {
                showToast('Quantidade maior que o estoque disponível', 'error');
                $(this).val(estoque); // Ajusta para o máximo disponível
            }
            
            $('#adicionarProduto').prop('disabled', quantidade > estoque);
        }
    });

    // Quando mudar o tipo de operação
    $('#tipoOperacao').on('change', function() {
        const tipo = $(this).val();
        $('#btnFinalizar').text(tipo === 'venda' ? 'Finalizar Venda' : 'Realizar Prova');
        $('#operacaoInfo').text(tipo === 'venda' ? 'Venda' : 'Prova');
        
        // Mostra/esconde método de pagamento baseado no tipo
        if (tipo === 'prova') {
            $('#metodoPagamentoContainer').hide();
        } else {
            $('#metodoPagamentoContainer').show();
        }
        
        atualizarBotaoFinalizar();
    });

    // Controle do método de pagamento
    $('#metodoPagamento').change(function() {
        const metodo = $(this).val();
        
        // Esconde o container de dinheiro/troco
        $('#dinheiroContainer').hide();
        $('#trocoContainer').hide();
        $('#crediarioInfo').hide();
        
        // Se for dinheiro, mostra o campo de valor recebido
        if (metodo === 'dinheiro') {
            $('#dinheiroContainer').show();
            $('#valorRecebido').val('');
            $('#trocoValue').text('R$ 0,00');
        }
        
        // Se for crediário, exige cliente e mostra aviso
        if (metodo === 'crediario') {
            if (!$('#cliente').val()) {
                showToast('Para Crediário é necessário selecionar um cliente', 'warning');
                $(this).val('');
                return;
            }
            $('#crediarioInfo').show();
        }
        
        atualizarBotaoFinalizar();
    });

    // Máscara e cálculo do troco
    $('#valorRecebido').on('input', function() {
        let valor = $(this).val().replace(/\D/g, '');
        valor = (parseFloat(valor) / 100).toFixed(2);
        $(this).val(valor);

        const totalVenda = parseFloat($('#totalValue').text().replace(/[^\d,]/g, '').replace(',', '.'));
        const valorRecebido = parseFloat(valor);

        if (!isNaN(valorRecebido) && valorRecebido >= totalVenda) {
            const troco = valorRecebido - totalVenda;
            $('#trocoValue').text(`R$ ${troco.toFixed(2).replace('.', ',')}`);
            $('#trocoContainer').show();
        } else {
            $('#trocoContainer').hide();
        }
        
        atualizarBotaoFinalizar();
    });

    // Quando selecionar um cliente
    $('#cliente').on('change', function() {
        const clienteNome = $(this).find('option:selected').text();
        $('#clienteInfo').text(clienteNome || 'Selecione um cliente');
    });

    // Adicionar produto ao carrinho
    $('#adicionarProduto').on('click', function() {
        const produtoId = $('#produto').val();
        const produtoNome = $('#produto option:selected').text();
        const tamanho = $('#tamanho').val();
        const quantidade = parseInt($('#quantidade').val());

        $.get('actions/get_produto_valor_pdv.php', { id: produtoId }, function(data) {
            const response = typeof data === 'string' ? JSON.parse(data) : data;
            if (response.success && response.data) {
                const valor = parseFloat(response.data.valor);
                const total = valor * quantidade;

                // Adiciona ao carrinho
                carrinho.push({
                    id: produtoId,
                    nome: produtoNome,
                    tamanho: tamanho,
                    quantidade: quantidade,
                    valor: valor,
                    total: total
                });

                // Atualiza o cupom
                const itemHtml = `
                    <div class="cupom-item" data-produto-id="${produtoId}" data-tamanho="${tamanho}" data-quantidade="${quantidade}" data-valor="${valor}">
                        <div class="item-nome">${produtoNome}</div>
                        <div class="item-detalhes">
                            <span>${quantidade}x ${tamanho} @ R$ ${valor.toFixed(2)}</span>
                            <span>R$ ${total.toFixed(2)}</span>
                        </div>
                    </div>
                `;
                $('#cupomItems').append(itemHtml);

                // Atualiza o total
                totalGeral += total;
                $('#totalValue').text(`R$ ${totalGeral.toFixed(2)}`);
                
                atualizarTotal();
                
                // Limpa a seleção
                $('#produto, #tamanho').val('').trigger('change');
                $('#quantidade').val(1);
                $('#adicionarProduto').prop('disabled', true);
            } else {
                showToast('Erro ao obter valor do produto', 'error');
            }
        }).fail(function() {
            showToast('Erro ao obter valor do produto', 'error');
        });
    });

    // Função para verificar se pode finalizar a venda
    function atualizarBotaoFinalizar() {
        const tipo = $('#tipoOperacao').val();
        const metodo = $('#metodoPagamento').val();
        const items = $('#cupomItems .cupom-item').length;
        let podeFinalizarPagamento = true;

        if (tipo === 'venda') {
            if (!metodo) {
                podeFinalizarPagamento = false;
            } else if (metodo === 'dinheiro') {
                const valorRecebido = parseFloat($('#valorRecebido').val() || 0);
                const totalVenda = parseFloat($('#totalValue').text().replace(/[^\d,]/g, '').replace(',', '.'));
                if (isNaN(valorRecebido) || valorRecebido < totalVenda) {
                    podeFinalizarPagamento = false;
                }
            }
        }

        $('#btnFinalizar').prop('disabled', items === 0 || (tipo === 'venda' && !podeFinalizarPagamento));
    }

    // Atualiza o valor total
    function atualizarTotal() {
        let total = 0;
        carrinho.forEach(item => {
            total += item.valor * item.quantidade;
        });

        $('#totalValue').text(`R$ ${total.toFixed(2).replace('.', ',')}`);
        
        // Limpa campos de pagamento quando o total muda
        $('#metodoPagamento').val('');
        $('#dinheiroContainer').hide();
        $('#trocoContainer').hide();
        $('#valorRecebido').val('');
        
        atualizarBotaoFinalizar();
    }

    // Finalizar venda
    $('#btnFinalizar').click(function() {
        if ($(this).prop('disabled')) return;

        const items = carrinho.map(item => ({
            produto_id: item.id,
            tamanho: item.tamanho,
            quantidade: item.quantidade,
            valor_unitario: item.valor
        }));

        const dados = {
            cliente_id: $('#cliente').val() || null,
            tipo: $('#tipoOperacao').val(),
            items: items,
            metodo_pagamento: $('#tipoOperacao').val() === 'venda' ? $('#metodoPagamento').val() : null,
            valor_recebido: $('#metodoPagamento').val() === 'dinheiro' ? parseFloat($('#valorRecebido').val()) : null
        };

        console.log('Dados sendo enviados:', dados);

        $.ajax({
            url: 'actions/finalizar_venda.php',
            method: 'POST',
            contentType: 'application/json',
            data: JSON.stringify(dados),
            success: function(response) {
                if (response.success) {
                    showToast('Venda finalizada com sucesso!', 'success');
                    // Limpa o PDV
                    carrinho = [];
                    totalGeral = 0;
                    $('#cliente').val('').trigger('change');
                    $('#metodoPagamento').val('').trigger('change');
                    $('#valorRecebido').val('');
                    $('#troco').val('');
                    $('#observacoes').val('');
                    limparCupom();
                    atualizarCarrinho();
                    atualizarTotal();
                    atualizarBotaoFinalizar();
                } else {
                    showToast(response.message || 'Erro ao finalizar venda', 'error');
                }
            },
            error: function(xhr) {
                console.error('Erro:', xhr.responseText);
                showToast('Erro ao finalizar venda. Verifique o console para mais detalhes.', 'error');
            }
        });
    });

    // Cancelar operação
    $('#btnCancelar').on('click', function() {
        if (confirm('Tem certeza que deseja cancelar esta operação?')) {
            location.reload();
        }
    });

    function limparCupom() {
        carrinho = [];
        totalGeral = 0;
        $('#cupomItems').empty();
        $('#totalValue').text('R$ 0,00');
        $('#cliente, #produto, #tamanho').val('').trigger('change');
        $('#quantidade').val(1);
        $('#tipoOperacao').val('venda');
        $('#clienteInfo').text('Selecione um cliente');
        $('#operacaoInfo').text('Venda');
    }

    // Atualiza campos conforme método de pagamento
    $('#metodoPagamento').change(function() {
        const metodo = $(this).val();
        const valorTotal = calcularTotal();

        // Reseta campos
        $('#divValorRecebido').hide();
        $('#divTroco').hide();
        $('#valorRecebido').val('');
        $('#troco').text('R$ 0,00');
        $('#crediarioInfo').hide();

        // Se for dinheiro, mostra campos de valor recebido e troco
        if (metodo === 'dinheiro') {
            $('#divValorRecebido').show();
            $('#divTroco').show();
            $('#valorRecebido').focus();
        }

        // Se for crediário, exige cliente e mostra aviso
        if (metodo === 'crediario') {
            if (!$('#cliente').val()) {
                showToast('Para Crediário é necessário selecionar um cliente', 'warning');
                $(this).val('');
                return;
            }
            $('#crediarioInfo').show();
        }

        // Atualiza validações do botão finalizar
        validarFinalizacao();
    });

    // Calcula troco quando valor recebido é alterado
    $('#valorRecebido').on('input', function() {
        const valorRecebido = parseFloat($(this).val()) || 0;
        const valorTotal = calcularTotal();
        const troco = valorRecebido - valorTotal;

        $('#troco').text(troco >= 0 ? 
            `R$ ${troco.toFixed(2)}` : 
            'Valor insuficiente'
        );

        // Atualiza validações do botão finalizar
        validarFinalizacao();
    });

    // Valida se pode finalizar a venda
    function validarFinalizacao() {
        const items = $('.cupom-item').length;
        const tipo = $('#tipoOperacao').val();
        const metodo = $('#metodoPagamento').val();
        const valorRecebido = parseFloat($('#valorRecebido').val()) || 0;
        const valorTotal = calcularTotal();
        let podeFinalizarPagamento = true;

        // Se for venda, valida método de pagamento
        if (tipo === 'venda') {
            if (!metodo) {
                podeFinalizarPagamento = false;
            } else if (metodo === 'dinheiro' && valorRecebido < valorTotal) {
                podeFinalizarPagamento = false;
            } else if (metodo === 'crediario' && !$('#cliente').val()) {
                podeFinalizarPagamento = false;
            }
        }

        // Habilita/desabilita botão finalizar
        $('#btnFinalizar').prop('disabled', 
            items === 0 || !podeFinalizarPagamento
        );
    }

    // Calcula o valor total
    function calcularTotal() {
        let total = 0;
        $('.cupom-item').each(function() {
            const valor = parseFloat($(this).data('valor'));
            const quantidade = parseInt($(this).data('quantidade'));
            total += valor * quantidade;
        });

        return total;
    }

    // Função para carregar prova no PDV
    async function carregarProva(provaId) {
        try {
            // Primeiro, carregamos os dados da prova
            const response = await $.ajax({
                url: 'actions/get_prova_para_pdv.php',
                method: 'GET',
                data: { prova_id: provaId },
                dataType: 'json'
            });
            
            if (response.success && response.data) {
                const prova = response.data.prova;
                const itens = response.data.itens;

                // Aguarda o carregamento dos clientes
                await new Promise((resolve) => {
                    const maxAttempts = 50;
                    let attempts = 0;
                    const checkCliente = setInterval(() => {
                        attempts++;
                        const clienteSelect = $('#cliente');
                        if (clienteSelect.find('option').length > 1) {
                            clearInterval(checkCliente);
                            clienteSelect.val(prova.cliente_id).trigger('change');
                            resolve();
                        } else if (attempts >= maxAttempts) {
                            clearInterval(checkCliente);
                            resolve();
                        }
                    }, 100);
                });

                // Seleciona operação como Venda
                $('#tipoOperacao').val('venda').trigger('change');
                
                // Limpa o carrinho atual
                carrinho = [];
                
                // Para cada item da prova
                for (const item of itens) {
                    carrinho.push({
                        id: item.produto_id,
                        nome: item.produto_nome,
                        tamanho: item.tamanho,
                        valor: parseFloat(item.valor_unitario),
                        quantidade: parseInt(item.quantidade),
                        total: parseFloat(item.valor_unitario) * parseInt(item.quantidade)
                    });
                }
                
                // Atualiza a exibição do carrinho
                atualizarCarrinho();
                atualizarTotal();
                atualizarBotaoFinalizar();

                // Finaliza a prova (devolve ao estoque e marca como convertida)
                $.ajax({
                    url: 'actions/finalizar_prova.php',
                    method: 'POST',
                    data: { prova_id: provaId },
                    success: function(response) {
                        if (response.success) {
                            showToast('Prova carregada com sucesso!', 'success');
                        } else {
                            console.error('Erro ao finalizar prova:', response.message);
                            showToast('Prova carregada, mas houve um erro ao finalizá-la: ' + response.message, 'warning');
                        }
                    },
                    error: function(xhr) {
                        console.error('Erro ao finalizar prova:', xhr.responseText);
                        showToast('Prova carregada, mas houve um erro ao finalizá-la', 'warning');
                    }
                });
            } else {
                showToast(response.message || 'Erro ao carregar dados da prova', 'error');
            }
        } catch (error) {
            console.error('Erro ao carregar prova:', error);
            showToast('Erro ao carregar prova. Verifique o console para mais detalhes.', 'error');
        }
    }

    // Função para atualizar o carrinho na interface
    function atualizarCarrinho() {
        const cupomItems = $('#cupomItems');
        cupomItems.empty();
        
        carrinho.forEach((item, index) => {
            const itemHtml = `
                <div class="cupom-item" data-index="${index}">
                    <div class="item-nome">${item.nome}</div>
                    <div class="item-detalhes">
                        <span>${item.quantidade}x ${item.tamanho} @ R$ ${item.valor.toFixed(2)}</span>
                        <span>R$ ${(item.valor * item.quantidade).toFixed(2)}</span>
                    </div>
                    <button type="button" class="btn btn-link text-danger btn-sm p-0 ms-2" onclick="removerItem(${index})">
                        <i class="bi bi-trash"></i>
                    </button>
                </div>
            `;
            cupomItems.append(itemHtml);
        });
        
        atualizarTotal();
    }

    // Verifica se há uma prova para carregar na URL
    const urlParams = new URLSearchParams(window.location.search);
    const provaId = urlParams.get('prova_id');
    const tipo = urlParams.get('tipo');
    
    if (provaId && tipo === 'venda') {
        console.log('Detectado prova_id na URL:', provaId);
        // Aguarda um momento para garantir que todos os componentes foram inicializados
        setTimeout(() => {
            console.log('Iniciando carregamento da prova após timeout');
            carregarProva(provaId);
        }, 2000); // Aumentei para 2 segundos para dar mais tempo de carregar
    }
}
