<?php
require_once '../includes/config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

try {
    // Recebe e valida os dados do formulário
    $id = filter_input(INPUT_POST, 'id', FILTER_SANITIZE_NUMBER_INT);
    $nome = filter_input(INPUT_POST, 'nome', FILTER_SANITIZE_STRING);
    $valor = filter_input(INPUT_POST, 'valor', FILTER_SANITIZE_STRING);
    $valor = str_replace(['R$', '.', ' '], '', $valor);
    $valor = str_replace(',', '.', $valor);
    $categoria_id = filter_input(INPUT_POST, 'categoria_id', FILTER_SANITIZE_NUMBER_INT);
    $cor = filter_input(INPUT_POST, 'cor', FILTER_SANITIZE_STRING);

    // Validação dos campos obrigatórios
    if (!$nome || !$valor || !$categoria_id || !$cor) {
        http_response_code(400);
        echo json_encode(['error' => 'Todos os campos são obrigatórios']);
        exit;
    }

    $pdo = getConnection();
    $pdo->beginTransaction();

    try {
        // Processa o upload da imagem
        $imagem = null;
        if (isset($_FILES['imagem']) && $_FILES['imagem']['error'] === UPLOAD_ERR_OK) {
            $uploadDir = '../uploads/produtos/';
            if (!file_exists($uploadDir)) {
                mkdir($uploadDir, 0777, true);
            }

            $fileName = uniqid() . '_' . basename($_FILES['imagem']['name']);
            $uploadFile = $uploadDir . $fileName;

            if (!move_uploaded_file($_FILES['imagem']['tmp_name'], $uploadFile)) {
                throw new Exception('Erro ao fazer upload da imagem');
            }

            $imagem = $fileName;
            
            // Se for edição, exclui a imagem antiga
            if ($id) {
                $stmt = $pdo->prepare("SELECT imagem FROM produtos WHERE id = ?");
                $stmt->execute([$id]);
                $imagemAntiga = $stmt->fetchColumn();
                
                if ($imagemAntiga && file_exists($uploadDir . $imagemAntiga)) {
                    unlink($uploadDir . $imagemAntiga);
                }
            }
        }

        if ($id) {
            // Atualiza o produto
            $sql = "UPDATE produtos SET 
                    nome = :nome, 
                    valor = :valor, 
                    categoria_id = :categoria_id, 
                    cor = :cor";
            
            if ($imagem !== null) {
                $sql .= ", imagem = :imagem";
            }
            
            $sql .= " WHERE id = :id";
            
            $stmt = $pdo->prepare($sql);
            
            $params = [
                ':nome' => $nome,
                ':valor' => $valor,
                ':categoria_id' => $categoria_id,
                ':cor' => $cor,
                ':id' => $id
            ];
            
            if ($imagem !== null) {
                $params[':imagem'] = $imagem;
            }
            
            $stmt->execute($params);
            $produtoId = $id;
            
            // Remove os tamanhos antigos
            $stmt = $pdo->prepare("DELETE FROM produto_tamanhos WHERE produto_id = ?");
            $stmt->execute([$id]);
        } else {
            // Insere novo produto
            $stmt = $pdo->prepare("
                INSERT INTO produtos (nome, valor, categoria_id, imagem, cor)
                VALUES (:nome, :valor, :categoria_id, :imagem, :cor)
            ");

            $stmt->execute([
                ':nome' => $nome,
                ':valor' => $valor,
                ':categoria_id' => $categoria_id,
                ':imagem' => $imagem,
                ':cor' => $cor
            ]);

            $produtoId = $pdo->lastInsertId();
        }

        // Salva os tamanhos e estoques
        $stmt = $pdo->prepare("
            INSERT INTO produto_tamanhos (produto_id, tamanho, estoque) 
            VALUES (?, ?, ?)
        ");

        foreach ($_POST['tamanhos'] as $i => $tamanho) {
            if (!empty($tamanho) && isset($_POST['estoques'][$i])) {
                $stmt->execute([
                    $produtoId,
                    $tamanho,
                    $_POST['estoques'][$i]
                ]);
            }
        }

        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => $id ? 'Produto atualizado com sucesso' : 'Produto cadastrado com sucesso'
        ]);

    } catch (Exception $e) {
        $pdo->rollBack();
        if (isset($imagem) && file_exists($uploadDir . $imagem)) {
            unlink($uploadDir . $imagem);
        }
        throw $e;
    }

} catch (Exception $e) {
    error_log("Erro ao salvar produto: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
