<?php
// Impede que erros do PHP sejam exibidos na saída
ini_set('display_errors', 0);
error_reporting(E_ALL);
ini_set('log_errors', 1);
ini_set('error_log', dirname(__DIR__) . '/logs/php_error.log');

require_once '../config/database.php';

header('Content-Type: application/json');
error_log("Iniciando salvamento de cliente...");

try {
    $database = new Database();
    $db = $database->getConnection();
    
    // Recebe os dados do formulário
    $id = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
    $nome = trim($_POST['nome'] ?? '');
    $data_nascimento = trim($_POST['data_nascimento'] ?? '');
    $endereco = trim($_POST['endereco'] ?? '');
    $numero = trim($_POST['numero'] ?? '');
    $bairro = trim($_POST['bairro'] ?? '');
    $cidade = trim($_POST['cidade'] ?? '');
    $telefone = trim($_POST['telefone'] ?? '');
    $instagram = trim($_POST['instagram'] ?? '');
    
    error_log("Dados recebidos: " . json_encode($_POST));
    
    // Validações básicas
    if (empty($nome)) {
        throw new Exception('O nome é obrigatório');
    }
    
    // Valida data de nascimento
    if (!empty($data_nascimento)) {
        $data = DateTime::createFromFormat('Y-m-d', $data_nascimento);
        if (!$data || $data->format('Y-m-d') !== $data_nascimento) {
            throw new Exception('Data de nascimento inválida');
        }
    }
    
    // Prepara os campos e valores para o SQL
    $campos = [
        'nome' => $nome,
        'data_nascimento' => $data_nascimento ?: null,
        'endereco' => $endereco ?: null,
        'numero' => $numero ?: null,
        'bairro' => $bairro ?: null,
        'cidade' => $cidade ?: null,
        'telefone' => $telefone ?: null,
        'instagram' => $instagram ?: null
    ];
    
    error_log("Campos preparados: " . json_encode($campos));
    
    if ($id) {
        // Atualização
        $campos['updated_at'] = date('Y-m-d H:i:s');
        $sets = [];
        foreach ($campos as $campo => $valor) {
            $sets[] = "{$campo} = ?";
        }
        $sql = "UPDATE clientes SET " . implode(', ', $sets) . " WHERE id = ?";
        $valores = array_values($campos);
        $valores[] = $id;
        
        error_log("SQL Update: $sql");
    } else {
        // Inserção
        $campos['created_at'] = date('Y-m-d H:i:s');
        $sql = "INSERT INTO clientes (" . implode(', ', array_keys($campos)) . ") 
                VALUES (" . str_repeat('?,', count($campos)-1) . "?)";
        $valores = array_values($campos);
        
        error_log("SQL Insert: $sql");
    }
    
    error_log("Valores: " . json_encode($valores));
    
    $stmt = $db->prepare($sql);
    $stmt->execute($valores);
    
    if (!$id) {
        $id = $db->lastInsertId();
    }
    
    error_log("Cliente salvo com sucesso. ID: $id");
    
    echo json_encode([
        'success' => true,
        'message' => 'Cliente salvo com sucesso!',
        'id' => $id
    ]);
    
} catch (PDOException $e) {
    error_log("Erro de banco de dados: " . $e->getMessage());
    error_log("SQL State: " . $e->getCode());
    error_log("Stack trace: " . $e->getTraceAsString());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao salvar cliente: ' . $e->getMessage()
    ]);
} catch (Exception $e) {
    error_log("Erro geral: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
