<?php
header('Content-Type: application/json');

require_once '../config/conexao.php';
require_once '../includes/functions.php';

try {
    $pdo = getConnection();
    $pdo->beginTransaction();

    // Recebe os dados do POST
    $dados = json_decode(file_get_contents('php://input'), true);
    
    if (!$dados) {
        throw new Exception('Dados inválidos');
    }

    // Log dos dados recebidos
    error_log('Dados recebidos: ' . print_r($dados, true));

    // Valida os dados necessários
    if (empty($dados['items'])) {
        throw new Exception('Nenhum item informado');
    }

    // Se for venda, valida o método de pagamento
    if ($dados['tipo'] === 'venda') {
        if (empty($dados['metodo_pagamento'])) {
            throw new Exception('Método de pagamento não informado');
        }

        // Se for dinheiro, valida o valor recebido
        if ($dados['metodo_pagamento'] === 'dinheiro' && 
            (!isset($dados['valor_recebido']) || $dados['valor_recebido'] <= 0)) {
            throw new Exception('Valor recebido inválido');
        }

        // Se for crediário, exige cliente
        if ($dados['metodo_pagamento'] === 'crediario' && empty($dados['cliente_id'])) {
            throw new Exception('Cliente não informado para venda no crediário');
        }
    }

    // Verifica estoque antes de prosseguir
    foreach ($dados['items'] as $item) {
        $stmt = $pdo->prepare("
            SELECT estoque as quantidade 
            FROM produto_tamanhos 
            WHERE produto_id = ? AND tamanho = ?
        ");
        $stmt->execute([$item['produto_id'], $item['tamanho']]);
        $estoque = $stmt->fetch(PDO::FETCH_ASSOC);
        
        error_log("Verificando estoque - Produto: {$item['produto_id']}, Tamanho: {$item['tamanho']}, Quantidade atual: " . ($estoque ? $estoque['quantidade'] : 'não encontrado'));
        
        if (!$estoque) {
            throw new Exception("Produto {$item['produto_id']} tamanho {$item['tamanho']} não encontrado no estoque");
        }
        
        if ($estoque['quantidade'] < $item['quantidade']) {
            throw new Exception("Estoque insuficiente para o produto {$item['produto_id']} tamanho {$item['tamanho']}. Disponível: {$estoque['quantidade']}, Solicitado: {$item['quantidade']}");
        }
    }

    // Calcula o valor total
    $valorTotal = 0;
    foreach ($dados['items'] as $item) {
        $valorTotal += $item['quantidade'] * $item['valor_unitario'];
    }

    // Define se afeta o caixa
    $afetaCaixa = true;
    if ($dados['tipo'] === 'prova' || ($dados['tipo'] === 'venda' && $dados['metodo_pagamento'] === 'crediario')) {
        $afetaCaixa = false;
    }

    // Define o cliente_id (pode ser null exceto para crediário)
    $clienteId = !empty($dados['cliente_id']) ? $dados['cliente_id'] : null;

    // Insere a operação
    $stmt = $pdo->prepare("
        INSERT INTO operacoes (
            cliente_id, 
            tipo, 
            valor_total,
            afeta_caixa,
            metodo_pagamento,
            valor_recebido,
            created_at,
            status
        ) VALUES (?, ?, ?, ?, ?, ?, NOW(), 'ativo')
    ");

    $stmt->execute([
        $clienteId,
        $dados['tipo'],
        $valorTotal,
        $afetaCaixa,
        $dados['tipo'] === 'venda' ? $dados['metodo_pagamento'] : null,
        $dados['metodo_pagamento'] === 'dinheiro' ? $dados['valor_recebido'] : null
    ]);

    $operacaoId = $pdo->lastInsertId();

    // Se for uma venda e afetar o caixa, registra na tabela vendas
    if ($dados['tipo'] === 'venda' && $afetaCaixa) {
        $stmt = $pdo->prepare("
            INSERT INTO vendas (
                operacao_id,
                usuario_id,
                cliente_id,
                valor_total,
                data_venda,
                status
            ) VALUES (?, ?, ?, ?, NOW(), 'concluida')
        ");
        $stmt->execute([
            $operacaoId,
            $_SESSION['usuario_id'] ?? null,
            $clienteId,
            $valorTotal
        ]);
    }

    // Insere os itens e atualiza o estoque
    foreach ($dados['items'] as $item) {
        // Insere o item
        $stmt = $pdo->prepare("
            INSERT INTO operacao_itens (
                operacao_id,
                produto_id,
                tamanho,
                quantidade,
                valor_unitario
            ) VALUES (?, ?, ?, ?, ?)
        ");

        $stmt->execute([
            $operacaoId,
            $item['produto_id'],
            $item['tamanho'],
            $item['quantidade'],
            $item['valor_unitario']
        ]);

        // Atualiza o estoque
        if ($dados['tipo'] === 'venda' || $dados['tipo'] === 'prova') {
            $stmt = $pdo->prepare("
                UPDATE produto_tamanhos 
                SET estoque = estoque - ? 
                WHERE produto_id = ? AND tamanho = ?
                AND estoque >= ?
            ");

            $stmt->execute([
                $item['quantidade'],
                $item['produto_id'],
                $item['tamanho'],
                $item['quantidade']
            ]);

            // Verifica se o estoque foi atualizado
            if ($stmt->rowCount() === 0) {
                throw new Exception("Erro ao atualizar estoque para produto {$item['produto_id']} tamanho {$item['tamanho']}. Verifique se há quantidade suficiente.");
            }
        }
    }

    $pdo->commit();

    echo json_encode([
        'success' => true,
        'message' => 'Operação finalizada com sucesso',
        'data' => [
            'id' => $operacaoId,
            'valor_total' => $valorTotal,
            'troco' => $dados['metodo_pagamento'] === 'dinheiro' ? 
                      $dados['valor_recebido'] - $valorTotal : 0
        ]
    ]);

} catch (Exception $e) {
    if (isset($pdo)) {
        $pdo->rollBack();
    }

    error_log('Erro na finalização da venda: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao finalizar operação: ' . $e->getMessage()
    ]);
}
