<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/../config/conexao.php';
require_once __DIR__ . '/../includes/functions.php';

header('Content-Type: application/json');

try {
    // Verifica se é uma requisição POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Método não permitido');
    }

    // Verifica se o ID foi enviado
    if (empty($_POST['id'])) {
        throw new Exception('ID do produto é obrigatório');
    }

    $id = intval($_POST['id']);
    if ($id <= 0) {
        throw new Exception('ID do produto inválido');
    }

    error_log("Tentando excluir produto ID: " . $id);

    $pdo = getConnection();
    $pdo->beginTransaction();

    try {
        // Busca o produto e verifica se existe
        $stmt = $pdo->prepare("SELECT id, imagem FROM produtos WHERE id = ? AND deleted_at IS NULL");
        $stmt->execute([$id]);
        $produto = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$produto) {
            throw new Exception('Produto não encontrado');
        }

        error_log("Produto encontrado: " . print_r($produto, true));

        // Remove os tamanhos
        $stmt = $pdo->prepare("DELETE FROM produto_tamanhos WHERE produto_id = ?");
        $stmt->execute([$id]);
        error_log("Tamanhos removidos");
        
        // Remove o produto (soft delete)
        $stmt = $pdo->prepare("UPDATE produtos SET deleted_at = NOW() WHERE id = ?");
        $stmt->execute([$id]);
        error_log("Produto marcado como excluído");
        
        // Remove a imagem se existir
        if (!empty($produto['imagem'])) {
            $imagemPath = __DIR__ . '/../uploads/produtos/' . $produto['imagem'];
            error_log("Tentando remover imagem: " . $imagemPath);
            if (file_exists($imagemPath)) {
                unlink($imagemPath);
                error_log("Imagem removida");
            } else {
                error_log("Imagem não encontrada em: " . $imagemPath);
            }
        }
        
        $pdo->commit();
        error_log("Transação commitada com sucesso");
        
        echo json_encode([
            'success' => true,
            'message' => 'Produto excluído com sucesso'
        ]);

    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Erro durante a transação: " . $e->getMessage());
        throw $e;
    }

} catch (Exception $e) {
    error_log("Erro ao excluir produto: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao excluir produto: ' . $e->getMessage()
    ]);
}
