<?php
header('Content-Type: application/json');
require_once '../config/database.php';

// Impede que erros do PHP sejam exibidos na saída
ini_set('display_errors', 0);
error_reporting(E_ALL);
ini_set('log_errors', 1);
ini_set('error_log', dirname(__DIR__) . '/logs/php_error.log');

try {
    if (!isset($_POST['id'])) {
        throw new Exception("ID do cliente não fornecido");
    }

    $id = $_POST['id'];
    error_log("=== Início da exclusão do cliente ID: " . $id . " ===");

    $database = new Database();
    $db = $database->getConnection();
    
    // Inicia uma transação
    $db->beginTransaction();
    error_log("Transação iniciada");
    
    try {
        // Debug: Verifica detalhes do cliente
        $query = "SELECT * FROM clientes WHERE id = :id AND deleted_at IS NULL";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        $cliente = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$cliente) {
            throw new Exception("Cliente não encontrado ou já foi excluído");
        }
        
        error_log("Detalhes do cliente: " . json_encode($cliente));

        // Verifica operações pendentes (não pagas)
        $query = "SELECT o.* FROM operacoes o 
                 WHERE o.cliente_id = :id 
                 AND o.deleted_at IS NULL 
                 AND o.status = 'pendente' 
                 AND (
                     o.valor_total > COALESCE((
                         SELECT SUM(valor) 
                         FROM pagamentos_parciais 
                         WHERE venda_id = o.id
                     ), 0)
                 )";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        $operacoes_pendentes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (count($operacoes_pendentes) > 0) {
            error_log("Operações pendentes encontradas: " . json_encode($operacoes_pendentes));
            throw new Exception("Não é possível excluir este cliente pois existem vendas com pagamento pendente.");
        }

        // Marca o cliente como excluído
        $query = "UPDATE clientes SET deleted_at = NOW() WHERE id = :id";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $id);
        
        if (!$stmt->execute()) {
            throw new Exception("Erro ao excluir o cliente");
        }

        // Marca todas as operações do cliente como excluídas
        $query = "UPDATE operacoes SET deleted_at = NOW() WHERE cliente_id = :id AND deleted_at IS NULL";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        $db->commit();
        error_log("Transação concluída com sucesso");
        
        echo json_encode([
            "success" => true,
            "message" => "Cliente excluído com sucesso"
        ]);
        
    } catch (Exception $e) {
        $db->rollBack();
        error_log("Erro durante a transação: " . $e->getMessage());
        throw $e;
    }
    
} catch (Exception $e) {
    error_log("Erro na exclusão do cliente: " . $e->getMessage());
    echo json_encode([
        "success" => false,
        "message" => $e->getMessage()
    ]);
}
